# -*- coding: utf-8 -*-


import zlib
from datetime import datetime, timedelta
from logging import Logger, getLogger
import collections

import ujson
from typing import List, Dict, Optional, Generator

from travel.avia.price_index.lib import flag_dependent_settings
from travel.avia.price_index.lib.currency_provider import CurrencyProvider, currency_provider
from travel.avia.price_index.lib.national_version_provider import NationalVersionProvider, national_version_provider
from travel.avia.price_index.lib.indexer import FareIndexBuilder, fare_index_builder
from travel.avia.price_index.lib.price_converter import PriceConverter, price_converter
from travel.avia.price_index.lib.rates_provider import RatesProvider, rates_provider
from travel.avia.price_index.models.flight import Flight
from travel.avia.price_index.models.query import IndexQuery
from travel.avia.price_index.models.result import Result
from travel.avia.library.proto.search_result.v1.result_pb2 import (
    Result as SearchResultProto,
    Flight as FlightProto,
    Variant as VariantProto,
)
from travel.avia.library.proto.common.v1.common_pb2 import PointType, NationalVersion, Baggage, SERVICE_CLASS_ECONOMY
from travel.proto.commons_pb2 import TDate


class ResultBuilder(object):
    def __init__(
        self, fare_index_builder, price_converter, rates_provider, currency_provider, national_version_provider, logger
    ):
        # type: (FareIndexBuilder, PriceConverter, RatesProvider, CurrencyProvider, NationalVersionProvider, Logger) -> None
        self._fare_index_builder = fare_index_builder
        self._price_converter = price_converter
        self._rates_provider = rates_provider
        self._currency_provider = currency_provider
        self._national_version_provider = national_version_provider
        self._logger = logger

    def build(self, index_query):
        # type: (IndexQuery) -> Result
        base_currency_id = self._rates_provider.get_base_currency_id(index_query.national_version_id)
        if base_currency_id is None:
            raise Exception('Can not fetch base currency id for [{}]'.format(index_query.national_version_id))
        rates_by_currency_id = self._rates_provider.get_rates_for(index_query.national_version_id)
        if rates_by_currency_id is None:
            raise Exception('Can not rates by currency id for [%d]', index_query.national_version_id)

        fare_list = index_query.raw_data['data']['variants']['fares']
        flight_by_key = {f['key']: self._build_flight(f) for f in index_query.raw_data['data']['reference']['flights']}
        variants = list(
            self._build_variants(
                index_query.national_version_id, fare_list, flight_by_key, base_currency_id, rates_by_currency_id
            )
        )

        best_variant = self._get_best_variant(variants)
        gzip_data = self._get_gzip_data(variants)

        return Result(
            national_version_id=index_query.national_version_id,
            from_id=index_query.from_id,
            to_id=index_query.to_id,
            adults_count=index_query.adults_count,
            children_count=index_query.children_count,
            infants_count=index_query.infants_count,
            forward_date=index_query.forward_date,
            backward_date=index_query.backward_date,
            base_value=best_variant['base_value'],
            value=best_variant['value'],
            currency_id=best_variant['currency_id'],
            created_at=datetime.now(),
            updated_at=datetime.now(),
            data=variants,
            gzip_data=gzip_data,
        )

    def build_from_proto(self, search_result):
        # type: (SearchResultProto) -> Optional[Result]
        national_version_code = NationalVersion.Name(search_result.national_version).split('_')[-1].lower()
        national_version_id = self._national_version_provider.get_by_code(national_version_code).pk
        base_currency_id = self._rates_provider.get_base_currency_id(national_version_id)
        if (
            search_result.point_from.type != PointType.POINT_TYPE_SETTLEMENT
            or search_result.point_to.type != PointType.POINT_TYPE_SETTLEMENT
        ):
            self._logger.info(
                '%s: From and to points must be of settlement type. from: %s to: %s',
                search_result.qid,
                search_result.point_from,
                search_result.point_to,
            )
            return None

        if search_result.service_class != SERVICE_CLASS_ECONOMY:
            self._logger.info('%s: Service class must be economy', search_result.qid)
            return None

        if base_currency_id is None:
            raise Exception(
                '{}: Can not fetch base currency id for [{}]'.format(search_result.qid, national_version_code)
            )
        rates_by_currency_id = self._rates_provider.get_rates_for(national_version_id)
        if rates_by_currency_id is None:
            raise Exception(
                '{}: Can not rates by currency id for [{}]'.format(search_result.qid, national_version_code)
            )

        flight_by_key = {
            key: self._build_flight_from_proto(search_result.flights[key]) for key in search_result.flights
        }
        fare_list = self._build_fares_from_proto(search_result.variants)
        variants = list(
            self._build_variants(national_version_id, fare_list, flight_by_key, base_currency_id, rates_by_currency_id)
        )

        best_variant = self._get_best_variant(variants)
        gzip_data = self._get_gzip_data(variants)

        return Result(
            national_version_id=national_version_id,
            from_id=search_result.point_from.id,
            to_id=search_result.point_to.id,
            adults_count=search_result.passengers.adults,
            children_count=search_result.passengers.children,
            infants_count=search_result.passengers.infants,
            forward_date=self._format_date(search_result.date_forward),
            backward_date=self._format_date(search_result.date_backward),
            base_value=best_variant['base_value'],
            value=best_variant['value'],
            currency_id=best_variant['currency_id'],
            created_at=datetime.now(),
            updated_at=datetime.now(),
            data=variants,
            gzip_data=gzip_data,
            qid=search_result.qid,
        )

    @staticmethod
    def _build_fares_from_proto(variants):
        # type: (List[VariantProto]) -> List[dict]
        by_route = collections.defaultdict(list)
        for v in variants:
            by_route[((f.flight_key for f in v.forward), (f.flight_key for f in v.backward))].append(
                {
                    'baggage': [[f.baggage for f in v.forward], [f.baggage for f in v.backward]],
                    'tariff': {'currency': v.price.currency, 'value': v.price.value},
                }
            )

        return [
            {
                'route': [list(r[0]), list(r[1])],
                'prices': prices,
            }
            for r, prices in by_route.items()
        ]

    def _get_gzip_data(self, variants):
        json_data = ujson.dumps(variants)
        gzip_data = zlib.compress(json_data.encode())
        self._logger.info('raw size: %d, gzip_data: %d', len(json_data), len(gzip_data))
        return gzip_data

    @staticmethod
    def _get_best_variant(variants):
        variants.sort(key=lambda v: v['base_value'])

        return (
            variants[0]
            if variants
            else {
                'base_value': None,
                'value': None,
                'currency_id': None,
            }
        )

    @staticmethod
    def _format_date(date_proto):
        # type: (TDate) -> Optional[basestring]
        if not date_proto or (date_proto.Year, date_proto.Month, date_proto.Day) == (0, 0, 0):
            return None
        return datetime(year=date_proto.Year, month=date_proto.Month, day=date_proto.Day).strftime('%Y-%m-%d')

    def _build_flight(self, raw_flight):
        arrival_local = datetime.strptime(raw_flight['arrival']['local'], '%Y-%m-%dT%H:%M:%S')
        arrival_utc = arrival_local + timedelta(minutes=raw_flight['arrival']['offset'])
        departure_local = datetime.strptime(raw_flight['departure']['local'], '%Y-%m-%dT%H:%M:%S')
        departure_utc = departure_local + timedelta(minutes=raw_flight['departure']['offset'])

        return Flight(
            departure=departure_local,
            arrival=arrival_local,
            departure_utc=departure_utc,
            arrival_utc=arrival_utc,
            airline_id=raw_flight['company'],
            from_id=raw_flight['from'],
            to_id=raw_flight['to'],
            number=raw_flight['number'],
        )

    def _build_flight_from_proto(self, flight_proto):
        # type: (FlightProto) -> Flight
        arrival_local = datetime.strptime(flight_proto.local_arrival, '%Y-%m-%dT%H:%M:%S')
        arrival_utc = datetime.strptime(flight_proto.utc_arrival, '%Y-%m-%dT%H:%M:%S')
        departure_local = datetime.strptime(flight_proto.local_departure, '%Y-%m-%dT%H:%M:%S')
        departure_utc = datetime.strptime(flight_proto.utc_departure, '%Y-%m-%dT%H:%M:%S')

        return Flight(
            departure=departure_local,
            arrival=arrival_local,
            departure_utc=departure_utc,
            arrival_utc=arrival_utc,
            airline_id=flight_proto.company_id,
            from_id=flight_proto.station_from_id,
            to_id=flight_proto.station_to_id,
            number=flight_proto.number,
        )

    def _build_variant(self, index, has_baggage, price):
        airport_index = index.airport_index
        transfer_index = index.transfer_index
        time_index = index.time_index

        return {
            'has_baggage': has_baggage,
            'base_value': price[0],
            'value': price[1],
            'currency_id': price[2],
            'airlines': list(index.airline_index),
            'forward_departure_airport': airport_index.forward.departure,
            'forward_transfer_airports': list(airport_index.forward.transfer),
            'forward_arrival_airport': airport_index.forward.arrival,
            'backward_departure_airport': airport_index.backward.departure,
            'backward_transfer_airports': list(airport_index.backward.transfer),
            'backward_arrival_airport': airport_index.backward.arrival,
            'forward_departure_time_type': time_index.forward_departure,
            'forward_arrival_time_type': time_index.forward_arrival,
            'backward_departure_time_type': time_index.backward_departure,
            'backward_arrival_time_type': time_index.backward_arrival,
            'count_transfer': transfer_index.count,
            'duration_transfer': transfer_index.duration,
            'has_airport_change': transfer_index.has_airport_change,
            'has_night_transfer': transfer_index.has_night_transfer,
        }

    def _has_baggage(self, price):
        for b in price['baggage'][0] + price['baggage'][1]:
            if b is None:
                return False
            if isinstance(b, str) and b.startswith('0') or isinstance(b, Baggage) and not b.included:
                return False
        return True

    def _get_min_price_from(self, national_version_id, fare, base_currency_id, rate_by_currency_id):
        with_baggage = []
        without_baggage = []

        for p in fare['prices']:
            currency_code = p['tariff']['currency']
            value = p['tariff']['value']

            currency = self._currency_provider.get_by_code(currency_code, national_version_id)
            if currency is None:
                continue
            currency_id = currency.pk

            base_value = self._price_converter.convert_to_base_currency_id(
                value, currency_id, base_currency_id, rate_by_currency_id
            )

            if base_value is None:
                continue

            price = (base_value, value, currency_id)

            if self._has_baggage(p):
                with_baggage.append(price)
            else:
                without_baggage.append(price)

        return (
            min(without_baggage, key=lambda x: x[0]) if without_baggage else None,
            min(with_baggage, key=lambda x: x[0]) if with_baggage else None,
        )

    def _build_variants(self, national_version_id, fares, flight_by_key, base_currency_id, rate_by_currency_id):
        # type: (int, List[dict], Dict[str, Flight], int, Dict[int, float]) -> Generator[dict]
        for f in fares:
            price_without_baggage, price_with_baggage = self._get_min_price_from(
                national_version_id=national_version_id,
                fare=f,
                base_currency_id=base_currency_id,
                rate_by_currency_id=rate_by_currency_id,
            )

            route = f['route']

            if len(route[0]) == 0:
                continue

            if not flag_dependent_settings.disable_stops_filter():
                if len(route[0]) > 3 or len(route[1]) > 3:
                    continue

            forward_flights = tuple(flight_by_key[f] for f in route[0])

            backward_flights = tuple(flight_by_key[f] for f in route[1])

            fare_index = self._fare_index_builder.index(forward_flights, backward_flights)

            if price_without_baggage:
                yield self._build_variant(fare_index, False, price_without_baggage)
            if price_with_baggage:
                yield self._build_variant(fare_index, True, price_with_baggage)


result_builder = ResultBuilder(
    logger=getLogger(__name__),
    fare_index_builder=fare_index_builder,
    price_converter=price_converter,
    rates_provider=rates_provider,
    currency_provider=currency_provider,
    national_version_provider=national_version_provider,
)
