# coding=utf-8
from typing import List, Generator
from collections import Counter
from multiprocessing import Pool
from contextlib import closing
import logging

from travel.library.python.logbroker.client import LogbrokerReader, DEFAULT_DC_ENDPOINTS
from travel.avia.library.proto.search_result.v1.result_pb2 import Result as SearchResultProto

from travel.avia.price_index.lib.result_builder import result_builder

logger = logging.getLogger(__name__)


class SearchResultsConsumer(LogbrokerReader):
    def __init__(self, consumer_id, topic_id, silent=False, token='', timeout=None, endpoints=DEFAULT_DC_ENDPOINTS):
        super(SearchResultsConsumer, self).__init__(
            consumer_id,
            [topic_id],
            silent=silent,
            token=token,
            timeout=timeout,
            endpoints=endpoints,
            balance_partition_now=True,
        )

    def read_batch(self, retry_on_timeout=False, proto_parsing_pool_size=10):
        # type: (bool, int) -> Generator[List[SearchResultProto]]
        with closing(Pool(processes=proto_parsing_pool_size)) as pool:
            while True:
                for batch in super(SearchResultsConsumer, self).read_batch(retry_on_timeout=retry_on_timeout):
                    messages = list(batch)
                    search_results = pool.map(_parse_proto_message, messages)

                    received_by_endpoint = Counter((m.endpoint for m in messages))
                    if received_by_endpoint:
                        for endpoint, counter in received_by_endpoint.items():
                            logger.info('received %d messages from %s', counter, endpoint)
                    else:
                        logger.info('received 0 messages')

                    yield search_results


def _parse_proto_message(message):
    search_result_proto = SearchResultProto()
    try:
        search_result_proto.ParseFromString(message.data)
    except Exception:
        logger.exception('Failed to parse proto message')
        return None

    try:
        return result_builder.build_from_proto(search_result_proto)
    except Exception:
        logger.exception('Failed to build result from parsed proto message')
        return None
