from datetime import datetime, timedelta
from logging import Logger, getLogger
from typing import Any

from marshmallow import ValidationError
from sqlalchemy import func
from sqlalchemy.orm import Session

from travel.avia.price_index.db_models.result import Result
from travel.avia.price_index.models.dump_form import DumpForm
from travel.avia.price_index.schemas.dump_form import DumpFormSchema
from travel.avia.price_index.views.helpers import BadRequest


class DumpView(object):
    def __init__(self, logger: Logger):
        self._logger = logger

    def parse_form(self, params):
        try:
            result = DumpFormSchema().load(params)
        except ValidationError as e:
            raise BadRequest('Can not parse search form') from e

        return result

    def process(self, session: Session, form: DumpForm) -> list[dict[str, Any]]:
        now = datetime.now().date()

        filters = [
            Result.adults_count == form.adults_count,
            Result.children_count == form.children_count,
            Result.infants_count == form.infants_count,
            Result.forward_date >= now,
            Result.forward_date <= now + timedelta(days=form.delta),
        ]

        if form.from_id is not None:
            filters.append(Result.from_id == form.from_id)

        if form.to_id is not None:
            filters.append(Result.to_id == form.to_id)

        q = (
            session.query(
                Result.national_version_id,
                Result.from_id,
                Result.to_id,
                func.min(Result.base_value),
            )
            .filter(*filters)
            .group_by(
                Result.national_version_id,
                Result.from_id,
                Result.to_id,
            )
        )

        return [
            {'national_version_id': national_version_id, 'from_id': from_id, 'to_id': to_id, 'base_value': base_value}
            for national_version_id, from_id, to_id, base_value in q
        ]


dump_view = DumpView(logger=getLogger(__name__))
