import ujson
from logging import Logger, getLogger

from marshmallow import ValidationError

from travel.avia.price_index.lib.constants import NULL_DATE
from travel.avia.price_index.lib.price_searchers.top_directions_by_date_window_searher import (
    TopDirectionsByDateWindowSearcher,
    top_directions_by_date_windows_searcher,
)
from travel.avia.price_index.models.top_directions_by_date_window_request import (
    TopDirectionsByDateWindowRequest,
    Direction,
)
from travel.avia.price_index.lib.rates_provider import rates_provider
from travel.avia.price_index.lib.currency_provider import currency_provider
from travel.avia.price_index.schemas.top_directions_by_date_window_query import TopDirectionsByDateWindowQuery
from travel.avia.price_index.views.helpers import json_response, failed_json_response


class TopDirectionsPricesByDateWindowView(object):
    def __init__(self, top_direction_by_date_window_searcher, logger):
        # type: (TopDirectionsByDateWindowSearcher, Logger) -> None
        self._top_directions_by_date_window_searcher = top_direction_by_date_window_searcher
        self._logger = logger

    def process(self, params, body, query_source):
        try:
            return self._process(params, body, query_source)
        except Exception as e:
            self._logger.exception('IndexView is raised 500: %r', e)
            return failed_json_response()

    def _process(self, params, body, query_source):
        post_data = ujson.loads(body)
        try:
            form_data = TopDirectionsByDateWindowQuery().load(
                dict(national_version=params['national_version'], **post_data)
            )
        except ValidationError as e:
            return json_response(data=e.messages, status=400)
        self._logger.info("Form data %s", form_data)

        national_version_id = form_data['national_version_id']
        rows = self._top_directions_by_date_window_searcher.search(
            national_version_id=form_data['national_version_id'],
            request=TopDirectionsByDateWindowRequest(
                directions=[Direction(orig_id=d['from_id'], dest_id=d['to_id']) for d in form_data['directions']],
                forward_date=form_data['forward_date'],
                backward_date=form_data['backward_date'],
                window_size=form_data['window_size'],
                results_per_direction=form_data['results_per_direction'],
            ),
        )

        currency_id = rates_provider.get_base_currency_id(nv_id=national_version_id)
        currency_code = currency_provider.get_by_id(currency_id, national_version_id).code
        self._logger.info("Rows %r", rows)

        return json_response(data={'status': 'ok', 'data': self._serialize_rows(rows, currency_code)}, status=200)

    def _serialize_rows(self, rows, currency_code):
        result_rows = []
        for r in rows:
            result_row = {
                'from_id': r['from_id'],
                'to_id': r['to_id'],
                'forward_date': r['forward_date'].isoformat(),
                'min_price': self._serialize_price(r['min_price'], currency_code),
            }
            if r['backward_date'] and r['backward_date'] != NULL_DATE:
                result_row['backward_date'] = r['backward_date'].isoformat()
            result_rows.append(result_row)
        return result_rows

    def _serialize_price(self, value, currency_code):
        if not value:
            return None
        return {'value': value, 'currency': currency_code}


top_direction_prices_by_date_window_view = TopDirectionsPricesByDateWindowView(
    top_direction_by_date_window_searcher=top_directions_by_date_windows_searcher, logger=getLogger(__name__)
)
