package main

import (
	"context"
	"fmt"
	"sync"
	"time"

	ghealth "google.golang.org/grpc/health"
	healthpb "google.golang.org/grpc/health/grpc_health_v1"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/maxprocs"
	"a.yandex-team.ru/library/go/yandex/tvm"
	"a.yandex-team.ru/library/go/yandex/tvm/cachedtvm"
	"a.yandex-team.ru/library/go/yandex/tvm/tvmtool"

	"a.yandex-team.ru/travel/avia/price_prediction/internal/app"
	checkpricehandler "a.yandex-team.ru/travel/avia/price_prediction/internal/checkprice/handler"
	"a.yandex-team.ru/travel/avia/price_prediction/internal/models"
	"a.yandex-team.ru/travel/avia/price_prediction/internal/service/checkprice"
	"a.yandex-team.ru/travel/library/go/configuration"
	grpcserver "a.yandex-team.ru/travel/library/go/grpcutil/server"
	"a.yandex-team.ru/travel/library/go/logging"
	"a.yandex-team.ru/travel/library/go/metrics"
	metricserver "a.yandex-team.ru/travel/library/go/metrics/server"
	"a.yandex-team.ru/travel/library/go/tracing"
	tvmutil "a.yandex-team.ru/travel/library/go/tvm"
)

const serviceName = "price_prediction"

func main() {
	maxprocs.AdjustAuto()

	// setting up infrastructure
	ctx, ctxCancel := context.WithCancel(context.Background())
	config := configuration.NewDefaultConfitaLoader()
	err := config.Load(ctx, &app.Cfg)

	if err != nil {
		fmt.Println("can not load configuration:", err)
		ctxCancel()
		return
	}

	logger, err := logging.New(&app.Cfg.Logging)
	if err != nil {
		fmt.Println("failed to create logger, err:", err)
		ctxCancel()
		return
	}

	tracerCloser := tracing.InitializeDefaultTracer(serviceName)
	defer func() {
		err = tracerCloser.Close()
		if err != nil {
			logger.Error("tracer close error:", log.Error(err))
		}

		err = logger.L.Sync()
		if err != nil {
			fmt.Println("failed to close logger:", err)
			return
		}
		ctxCancel()
	}()

	var tvmClient tvm.Client
	tvmAllowedIds := tvmutil.TvmClientIDFromInt(app.Cfg.Tvm.WhiteList)
	if app.Cfg.Tvm.Enabled {
		tvmClient, err = tvmtool.NewDeployClient()
		if err != nil {
			fmt.Println("failed to create tvm client:", err)
		}
		tvmClient, err = cachedtvm.NewClient(tvmClient, cachedtvm.WithCheckServiceTicket(app.Cfg.Tvm.CacheTTL, app.Cfg.Tvm.CacheMaxItems))
		if err != nil {
			fmt.Println("failed to create cached tvm client:", err)
		}
	}

	rootRegistry := metrics.NewRegistryWithDeployTagsAndExplicitHost()
	// in order to make it possible to collect arbitrary application metrics through metrics.GlobalAppMetrics
	appMetrics := metrics.NewAppMetrics(rootRegistry.WithPrefix("app"))
	metrics.SetGlobalAppMetrics(appMetrics)
	metrics.RunPerfMetricsUpdater(rootRegistry, app.Cfg.Metrics.PerfMetricsRefreshInterval)

	pgClient, err := app.BuildPGClient(logger)
	if err != nil {
		logger.Error("failed to create PG client", log.Error(err))
		return
	}

	if err := models.MigrateAndInit(pgClient); err != nil {
		logger.Errorf(err.Error())
		return
	}

	variantsPriceStatsRepository, err := app.BuildVariantsPriceStatsRepository(pgClient, logger)
	if err != nil {
		logger.Error("failed to read DB content", log.Error(err))
		return
	}

	checkPriceService := checkprice.BuildService(logger, variantsPriceStatsRepository, app.Cfg.IsTesting())
	checkPriceHandler := checkpricehandler.NewGRPCCheckPriceHandler(logger, checkPriceService.CheckPrice)

	grpcServiceRegisterers := []grpcserver.ServiceRegisterer{
		checkPriceHandler.GetServiceRegisterer(),
	}

	healthServer := ghealth.NewServer()
	// The health-checking logic of a specific application should be implemented below
	go func() {
		for range time.Tick(2 * time.Second) {
			healthServer.SetServingStatus(serviceName, healthpb.HealthCheckResponse_SERVING)
		}
	}()

	// running server
	wg := sync.WaitGroup{}
	wg.Add(3)

	go func() {
		defer wg.Done()
		server := grpcserver.NewDefaultGrpcServerBuilder(
			app.Cfg.Grpc,
			grpcServiceRegisterers,
			logger,
			tvmClient,
			tvmAllowedIds,
			rootRegistry.WithPrefix("grpc"),
		).WithHealthServer(healthServer).Build()
		err = server.Run(context.Background())
		if err != nil {
			logger.Fatal("Error while starting Grpc server", log.Error(err))
		}
	}()

	go func() {
		defer wg.Done()
		err = metricserver.RunMetricsHTTPServer(context.Background(), app.Cfg.Metrics, logger, rootRegistry)
		if err != nil {
			logger.Fatal("Error while starting metrics server", log.Error(err))
		}
	}()

	go func() {
		defer wg.Done()
		variantsPriceStatsRepository.RunUpdater()
		logger.Fatal("Updater exited")
	}()

	logger.Info("Started")
	wg.Wait()
}
