package checkprice

import (
	pb "a.yandex-team.ru/travel/avia/price_prediction/api/v1"
	"context"
	"fmt"
	"strconv"
	"strings"
	"time"
)

type CheckPriceRequest struct {
	PointFromType      string
	PointFromID        uint
	PointToType        string
	PointToID          uint
	RouteUID           string
	DepartureWeekday   uint8
	DaysToFlightBucket uint
	AdultSeats         uint
	ChildrenSeats      uint
	InfantSeats        uint
	Price              float64

	LocalDeparture time.Time // Need this only for log bad and good prices
}

type PriceCategory int

const (
	PriceCategoryUnknown PriceCategory = iota
	PriceCategoryGood
	PriceCategoryBad
)

type PriceChecker func(context.Context, CheckPriceRequest) PriceCategory

func NewCheckPriceRequest(req *pb.TCheckPriceReq) (*CheckPriceRequest, error) {
	pointFromType, pointFromID, err := getPointTypeAndID(req.PointFromKey)
	if err != nil {
		return nil, fmt.Errorf("invalid PointFromKey '%v'", req.PointFromKey)
	}

	pointToType, pointToID, err := getPointTypeAndID(req.PointToKey)
	if err != nil {
		return nil, fmt.Errorf("invalid PointToKey '%v'", req.PointToKey)
	}

	if req.AdultSeats+req.ChildrenSeats+req.InfantSeats == 0 {
		return nil, fmt.Errorf("invalid Seats (%v, %v, %v)", req.AdultSeats, req.ChildrenSeats, req.InfantSeats)
	}

	routes := strings.TrimSpace(req.Routes)
	if len(routes) == 0 {
		return nil, fmt.Errorf("invalid Routes '%v'", req.Routes)
	}

	// При генерации данных считаем дельту в часах и делим на 24, здесь делаем так же
	daysToFlight := float64(req.LocalDeparture-time.Now().Unix()) / 60. / 60. / 24.
	if daysToFlight < 0 {
		return nil, fmt.Errorf("invalid LocalDeparture '%v'", req.LocalDeparture)
	}

	localDepartureDT := time.Unix(req.LocalDeparture, 0)
	// в данных в базе значение из python - Monday is 0 and Sunday is 6
	// в go Weekday - Sunday = 0
	departureWeekday := uint8((localDepartureDT.Weekday() + 6) % 7)

	checkPriceRequest := &CheckPriceRequest{
		PointFromType:      pointFromType,
		PointFromID:        pointFromID,
		PointToType:        pointToType,
		PointToID:          pointToID,
		RouteUID:           routes,
		DepartureWeekday:   departureWeekday,
		DaysToFlightBucket: getDaysToFlightBucket(int(daysToFlight)),
		AdultSeats:         uint(req.AdultSeats),
		ChildrenSeats:      uint(req.ChildrenSeats),
		InfantSeats:        uint(req.InfantSeats),
		Price:              req.Price,

		LocalDeparture: localDepartureDT,
	}
	return checkPriceRequest, nil
}

func getPointTypeAndID(pointKey string) (string, uint, error) {
	if len(pointKey) < 2 {
		return "", 0, fmt.Errorf("too few symbols in pointKey")
	}
	pointType := pointKey[:1]
	if !isValidPointType(pointType) {
		return "", 0, fmt.Errorf("invalid point type")
	}
	pointID, err := strconv.Atoi(pointKey[1:])
	return pointType, uint(pointID), err
}

func getDaysToFlightBucket(daysToFlight int) uint {
	if daysToFlight <= 1 {
		return 1
	} else if daysToFlight < 4 {
		return 2
	} else if daysToFlight < 8 {
		return 4
	} else if daysToFlight < 16 {
		return 8
	} else if daysToFlight < 32 {
		return 16
	} else if daysToFlight < 64 {
		return 32
	} else {
		return 64
	}
}

func isValidPointType(pointType string) bool {
	switch pointType {
	case
		"c",
		"s":
		return true
	}
	return false
}
