package checkprice

import (
	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/travel/avia/price_prediction/internal/checkprice"
	"a.yandex-team.ru/travel/avia/price_prediction/internal/models"
	"context"
	"fmt"
)

const qDeltaPercent = 3.0

type PriceCheckerInterface interface {
	CheckPrice(context.Context, checkprice.CheckPriceRequest) checkprice.PriceCategory
}

type VariantsPriceStatsRepository interface {
	GetByKey(priceStatKey *models.PriceStatKey) (*models.VariantPriceStat, bool)
}

type Service struct {
	logger                       log.Logger
	variantsPriceStatsRepository VariantsPriceStatsRepository
}

func (s *Service) CheckPrice(ctx context.Context, request checkprice.CheckPriceRequest) (result checkprice.PriceCategory) {
	s.logger.Debugf("CheckPriceService.CheckPrice called %+v", request)
	key := models.PriceStatKey{
		PointFromType:    request.PointFromType,
		PointFromID:      request.PointFromID,
		PointToType:      request.PointToType,
		PointToID:        request.PointToID,
		RouteUID:         request.RouteUID,
		DepartureWeekday: request.DepartureWeekday,
		DaysToFlight:     request.DaysToFlightBucket,
	}
	ps, exists := s.variantsPriceStatsRepository.GetByKey(&key)
	if !exists {
		writeCheckPriceMetric(predictionCategoryNotFound)
		return checkprice.PriceCategoryUnknown
	}
	q33 := ps.Q33 * (1. - qDeltaPercent/100.) * (1.0*float64(request.AdultSeats) + 0.6*float64(request.ChildrenSeats) + 0.1*float64(request.InfantSeats))
	q67 := ps.Q67 * (1. + qDeltaPercent/100.) * (1.0*float64(request.AdultSeats) + 0.6*float64(request.ChildrenSeats) + 0.1*float64(request.InfantSeats))
	if request.Price <= q33 {
		writeCheckPriceMetric(predictionCategoryGood)
		logCheckPriceResult(s.logger, "good price", request)
		return checkprice.PriceCategoryGood
	} else if request.Price >= q67 {
		writeCheckPriceMetric(predictionCategoryBad)
		logCheckPriceResult(s.logger, "bad price", request)
		return checkprice.PriceCategoryBad
	} else {
		writeCheckPriceMetric(predictionCategoryUnknown)
		return checkprice.PriceCategoryUnknown
	}
}

func newService(l log.Logger, r VariantsPriceStatsRepository) *Service {
	return &Service{logger: l, variantsPriceStatsRepository: r}
}

func BuildService(l log.Logger, r VariantsPriceStatsRepository, isTesting bool) PriceCheckerInterface {
	if isTesting {
		return newServiceInTesting(l, r)
	} else {
		return newService(l, r)
	}
}

func logCheckPriceResult(l log.Logger, msg string, request checkprice.CheckPriceRequest) {
	l.Info(msg,
		log.String("PointFromType", request.PointFromType),
		log.UInt("PointFromID", request.PointFromID),
		log.String("PointToType", request.PointToType),
		log.UInt("PointToID", request.PointToID),
		log.String("When", request.LocalDeparture.Format("2006-01-02")),
		log.String("Route", request.RouteUID),
		log.Float64("Price", request.Price),
		log.String("Passengers", fmt.Sprintf("%d_%d_%d", request.AdultSeats, request.ChildrenSeats, request.InfantSeats)),
	)
}
