# -*- coding: utf-8 -*-
from typing import Optional, Callable, Any, Iterator

import time
from contextlib2 import contextmanager, suppress
from selenium.common.exceptions import NoSuchElementException

from travel.avia.revise.extractor.error import TicketChangedException
from travel.avia.revise.extractor.report import DefaultScreenShotMaker
from travel.avia.revise.utils.helpers import catch_exc


class BasePartnerExtractor(object):
    """
    OLD_STYLE_EXTRACTOR: использовать для уже реализованного функтора extractor
    """

    IMPLICITLY_WAIT_TIME = 40
    SLEEP_BEFORE_REVIEW = 5
    DEBUG = False
    OLD_STYLE_EXTRACTOR: Optional[Callable[[Any, Any, Any, Any], Iterator[Any]]] = None
    # some partners show either price or warning, and we check warning only after we failed to find price.
    # But other partners show both, so even if there is the price we still have to look for the warning
    ALWAYS_CHECK_PRICE_CHANGE = False

    def __init__(self, driver, variant_params, url, post):
        self.driver = driver
        self._screenshot_maker = DefaultScreenShotMaker(driver=driver)
        self.variant_params = variant_params
        self.url = url
        self.post = post

    @classmethod
    def entrypoint(cls, driver, variant_params, url, post):
        return cls(driver, variant_params, url, post).extract()

    def modify_url(self, url):
        """
        По умолчанию не модифицируем url
        """
        return url

    def extract_price(self) -> Iterator[Any]:
        """
        Метод получения цены со страницы партнера.
        Существующую функцию можно присвоить в cls.OLD_STYLE_EXTRACTOR
        """
        raise NotImplementedError

    @staticmethod
    def ticket_changed():
        """Метод определения доступности билета"""
        return False

    def extract(self):
        self.driver.implicitly_wait(self.IMPLICITLY_WAIT_TIME)

        url = self.modify_url(self.url)
        yield 'url', url
        self.driver.get(url)

        time.sleep(self.SLEEP_BEFORE_REVIEW)

        with catch_exc(NoSuchElementException, TicketChangedException) as error:
            with self.check_ticket_changes():
                for item in self.get_extractor():
                    yield item

        if error.was_caught():
            with suppress(Exception):
                yield self._screenshot_maker.make('error', debug=self.DEBUG)
            error.reraise()

    @contextmanager
    def check_ticket_changes(self):
        with catch_exc(NoSuchElementException) as error:
            yield

        if error.was_caught() or self.ALWAYS_CHECK_PRICE_CHANGE:
            if self.ticket_changed():
                raise TicketChangedException('Found banner with a message about ticket changes')
            if error.was_caught():
                error.reraise()

    def get_extractor(self):
        if self.OLD_STYLE_EXTRACTOR:
            return self.get_unbound_extractor()(
                self.driver,
                self.variant_params,
                self.url,
                self.post,
            )
        return self.extract_price()

    @classmethod
    def get_unbound_extractor(cls) -> Optional[Callable[[Any, Any, Any, Any], Iterator[Any]]]:
        return cls.OLD_STYLE_EXTRACTOR
