# -*- coding: utf-8 -*-
import json
import logging
from collections.abc import Callable
from importlib import import_module
from typing import Union, Any, Optional

import sys
from pydantic import BaseModel, Field, validator
from selenium import webdriver

from travel.avia.revise.utils import helpers
from travel.avia.revise.extractor.report import ScreenShot, filter_dict
from travel.avia.revise.extractor.report import ExtractionError

logger = logging.getLogger(__name__)


class Price(BaseModel):
    value: float = Field(alias='price')
    currency: Optional[str]

    class Config:
        allow_population_by_field_name = True

    @validator('currency')
    def check_currency(cls, value: Optional[str]) -> Optional[str]:
        if value is not None and value not in helpers.CURRENCIES:
            raise ValueError(f"Currency {value} is unknown")
        return value


class ExtractedInfo(BaseModel):
    price: Price
    screenshots: list[ScreenShot]
    meta: Any


Extractor = Callable[[str, Optional[dict[str, Any]]], Union[ExtractedInfo, ExtractionError]]


def _get_module_extractor(partner: str) -> Callable[[webdriver.Remote, dict, str, Optional[dict[str, Any]]], Any]:
    try:
        return import_module(
            'travel.avia.revise.partners.%s' % partner
        ).extractor
    except Exception:
        logger.exception('No partner module: %r', partner)
        raise


def get_extractor(partner: str, driver: webdriver.Remote) -> Extractor:
    no_format_extractor = _get_module_extractor(partner)

    def extractor(url: str, post_params: Optional[dict[str, Any]] = None) -> Union[ExtractedInfo, ExtractionError]:
        report = []
        info = None
        screenshots = []
        try:
            for step, info in enumerate(no_format_extractor(driver, {}, url, post_params), 1):
                logger.debug('=== Step %s === info %r', step, info)
                if isinstance(info, ScreenShot):
                    screenshots.append(info)
                    report.append(filter_dict({
                        'kind': 'screenshot',
                        'key': step,
                        'name': info.name,
                        'error': info.error,
                    }))
                    continue
                item = info
                try:
                    json.dumps(item)
                except Exception:
                    item = repr(item)
                report.append(item)
            if info is None:
                raise RuntimeError('Extractor returned nothing')
        except Exception:
            return ExtractionError(exc_info=sys.exc_info(), screenshots=screenshots)
        return ExtractedInfo(price=Price.parse_obj(info), screenshots=screenshots, meta=report)

    return extractor
