# -*- coding: utf-8 -*-
from typing import Iterator, Any

import logging
import requests
import time
from selenium.common.exceptions import WebDriverException, TimeoutException, NoSuchElementException

from travel.avia.revise.extractor.base_partner_extractor import BasePartnerExtractor
from travel.avia.revise.utils.helpers import get_currency, parse_price, wait_for_element_xpath

logger = logging.getLogger(__name__)

FIREFOX_USERAGENT = 'Mozilla/5.0 (Windows NT 6.1; WOW64; rv:31.0) Gecko/20130401 Firefox/31.0'


class Extractor(BasePartnerExtractor):
    IMPLICITLY_WAIT_TIME = 1
    SLEEP_BEFORE_REVIEW = 1
    _PRICE_XPATH = '/html/body/div[1]/div[1]/div[2]/div/div[3]/div/div[2]/span/span'
    _WAIT_SCREEN = 60
    _COOKIE_SET_TIMEOUT = 2

    def ticket_changed(self):
        unavailability_warning_xpath = '//div[contains(text(),"Билеты распроданы")]'
        unavailability_warning = self.driver.find_elements_by_xpath(unavailability_warning_xpath)
        return bool(unavailability_warning)

    def extract_price(self) -> Iterator[Any]:
        logger.debug("Passed post params %s", self.post)
        response = requests.post(
            self.url,
            data=self.post,
            headers={'User-Agent': FIREFOX_USERAGENT},
            cookies=dict((str(cookie['name']), str(cookie['value'])) for cookie in self.driver.get_cookies())
        )
        logger.debug("Redirecting to %s", response.url)

        for cookie in response.cookies:
            cookie_dict = {'name': cookie.name, 'value': cookie.value, 'secure': cookie.secure}
            if cookie.expires:
                cookie_dict['expiry'] = cookie.expires
            if cookie.path_specified:
                cookie_dict['path'] = cookie.path

            # Taken from seleniumrequests library
            # Some drivers can not set cookies instantly
            start = time.time()
            while time.time() - start <= self._COOKIE_SET_TIMEOUT:
                try:
                    self.driver.add_cookie(cookie_dict)
                    break
                except WebDriverException:
                    pass
            else:
                raise TimeoutException('page took too long to load')

        self.driver.get(response.url)
        try:
            price_element = wait_for_element_xpath(self.driver, selector=self._PRICE_XPATH, delay=self._WAIT_SCREEN)
        except Exception as e:
            self._screenshot_maker.make(name='error', debug=self.DEBUG)
            raise NoSuchElementException from e
        yield self._screenshot_maker.make(name='first', debug=self.DEBUG)
        yield 'price_element', price_element
        yield 'price.text', price_element.text

        try:
            value = parse_price(price_element.text)
        except (ValueError, TypeError):
            yield self._screenshot_maker.make(name='price_parsing_error', debug=self.DEBUG)
            value = None
        currency = get_currency(price_element.text)

        yield {
            'price': value,
            'currency': currency,
        }


extractor = Extractor.entrypoint
