# -*- coding: utf-8 -*-
import json
import logging
from typing import Sequence

import requests

from travel.avia.revise import settings
from travel.avia.revise.extractor.choices import ResultChoices

log = logging.getLogger(__name__)

ReviseReport = dict


class ReviewWriter(object):
    def __init__(self, logger):
        self._logger = logger
        self.__rest_uri = 'https://%s/rest/' % settings.AVIA_STATS_REST_HOST

    def write_review(self, review, message_id):
        screenshots = review.pop('screenshots', [])

        try:
            review_id = self._write_review(review, message_id)
        except Exception:
            self._logger.exception('In write review')
        else:
            try:
                self._write_review_screenshots(review_id, screenshots, message_id)
            except Exception:
                self._logger.exception('In write review screenshots')

    def _write_review(self, review, message_id):
        serialized = json.dumps(
            dict(review),
            ensure_ascii=False
        )
        if isinstance(serialized, str):
            serialized = serialized.encode('utf-8')

        r = self._post('partner-review/', serialized, message_id)
        review_id = r.json()['id']

        return review_id

    def _write_review_screenshots(self, review_id: int, screenshots: Sequence[bytes], message_id: str):
        for idx, screenshot in enumerate(screenshots, start=1):
            serialized = json.dumps({
                'partner_review_id': review_id,
                'screenshot': screenshot.decode('ascii'),
                'idx': idx,
            })
            self._post('partner-review-screenshots/', serialized, message_id)

    def _post(self, view, data, message_id):
        url = self.__rest_uri + view
        try:
            r = requests.post(
                url,
                data=data,
                headers={
                    'content-type': 'application/json',
                    'Authorization': 'OAuth {}'.format(
                        settings.AVIA_STATS_OAUTH_TOKEN,
                    ),
                },
                timeout=settings.AVIA_STATS_REST_TIMEOUT,
                verify=False
            )

            if not r.ok:
                self._logger.error(
                    'Avia stats rest request fail, message id: %s: [%s] %s',
                    message_id, r.status_code, r.content.decode('utf-8')
                )
                r.raise_for_status()
            else:
                self._logger.info(
                    'Message (id: %s) was sent to stat with status Ok: %s %s,',
                    message_id, r.status_code, r.content.decode('utf-8')
                )
                return r
        except Exception as e:
            self._logger.exception(
                'Avia stats rest request fail(message id: %s): %r',
                message_id, e
            )
            raise


def get_result_status(difference):
    if abs(difference) <= 1:
        return ResultChoices.pass_choice, 'Price match'

    elif abs(difference) <= 10:
        return ResultChoices.problem_1_10, 'Price almost match'

    else:
        return ResultChoices.problem, 'Prices differ'


def get_prices_diffs(price_revise, price_to_compare):
    abs_difference = price_revise - price_to_compare
    rel_difference = round(100 * abs_difference / price_to_compare, 3)

    return abs_difference, rel_difference
