# -*- coding: utf-8 -*-
import re
import typing
from typing import Type, ContextManager

import sys
from contextlib2 import contextmanager
from itertools import chain
from selenium.common.exceptions import NoSuchElementException
from selenium.common.exceptions import TimeoutException
from selenium.webdriver import Remote
from selenium.webdriver.common.by import By
from selenium.webdriver.remote.webelement import WebElement
from selenium.webdriver.support import expected_conditions as ec
from selenium.webdriver.support.wait import WebDriverWait

ROUBLES = {'руб.', 'р.', 'р', 'руб', 'rur', 'p'}
CURRENCIES = frozenset((
    'AED', 'AFN', 'ALL', 'AMD', 'ANG', 'AOA', 'ARS', 'AUD', 'AWG', 'AZN', 'BAM', 'BBD', 'BDT', 'BGN', 'BHD', 'BIF',
    'BMD', 'BND', 'BOB', 'BRL', 'BSD', 'BTC', 'BTN', 'BWP', 'BYR', 'BZD', 'CAD', 'CDF', 'CHF', 'CLF', 'CLP', 'CNY',
    'COP', 'CRC', 'CUC', 'CUP', 'CVE', 'CZK', 'DJF', 'DKK', 'DOP', 'DZD', 'EEK', 'EGP', 'ERN', 'ETB', 'EUR', 'FJD',
    'FKP', 'GBP', 'GEL', 'GGP', 'GHS', 'GIP', 'GMD', 'GNF', 'GTQ', 'GYD', 'HKD', 'HNL', 'HRK', 'HTG', 'HUF', 'IDR',
    'ILS', 'IMP', 'INR', 'IQD', 'IRR', 'ISK', 'JEP', 'JMD', 'JOD', 'JPY', 'KES', 'KGS', 'KHR', 'KMF', 'KPW', 'KRW',
    'KWD', 'KYD', 'KZT', 'LAK', 'LBP', 'LKR', 'LRD', 'LSL', 'LTL', 'LVL', 'LYD', 'MAD', 'MDL', 'MGA', 'MKD', 'MMK',
    'MNT', 'MOP', 'MRO', 'MTL', 'MUR', 'MVR', 'MWK', 'MXN', 'MYR', 'MZN', 'NAD', 'NGN', 'NIO', 'NOK', 'NPR', 'NZD',
    'OMR', 'PAB', 'PEN', 'PGK', 'PHP', 'PKR', 'PLN', 'PYG', 'QAR', 'QUN', 'RON', 'RSD', 'RUB', 'RWF', 'SAR', 'SBD',
    'SCR', 'SDG', 'SEK', 'SGD', 'SHP', 'SLL', 'SOS', 'SRD', 'STD', 'SVC', 'SYP', 'SZL', 'THB', 'TJS', 'TMT', 'TND',
    'TOP', 'TRY', 'TTD', 'TWD', 'TZS', 'UAH', 'UGX', 'USD', 'UYU', 'UZS', 'VEF', 'VND', 'VUV', 'WST', 'XAF', 'XAG',
    'XAU', 'XCD', 'XDR', 'XOF', 'XPD', 'XPF', 'XPT', 'YER', 'ZAR', 'ZMK', 'ZMW', 'ZWL',
))


def grouped(iterable, n):
    for i in range(len(iterable) - n + 1):
        yield iterable[i: i + n]


def wait_for_element(driver: Remote, selector: str, delay: typing.Union[int, float]) -> WebElement:
    try:
        return WebDriverWait(driver, delay).until(
            ec.presence_of_element_located((By.CSS_SELECTOR, selector)))
    except TimeoutException:
        raise Exception("Loading took too much time!")


def wait_for_element_xpath(driver: Remote, selector: str, delay: typing.Union[int, float]) -> WebElement:
    try:
        return WebDriverWait(driver, delay).until(
            ec.presence_of_element_located((By.XPATH, selector)))
    except TimeoutException:
        raise Exception("Loading took too much time!")


def poll(retries, func, *args):
    """
    Just a wrapper for a retry action on webdriver functions.
    Pass in a function func, its arguemtns args.
    We shall try to get value from this function for `retries` times.
    If function func raises any exception but NoSuchElementException - we're raising it too.
    If func raises NoSuchElementException, we keep on trying

    :param int retries:
    :param func: webdriver function to retry
    :param args: arguments
    :return: Value, that is returned by a function
    :raises NoSuchElementException: if we could not find an element after retries
    :raises Exception: if func raises it
    """
    return poll_exc(retries, NoSuchElementException, func, *args)


def poll_exc(retries, exc, func, *args):
    """
    Just a wrapper for a retry action on webdriver functions.
    Pass in a function func, its arguemtns args and expected exception exc.
    We shall try to get value from this function for `retries` times.
    If function func raises any exception but NoSuchElementException - we're raising it too.
    If func raises NoSuchElementException, we keep on trying

    :param int retries:
    :param exc: class of exception that we should ignore while retrying
    :param func: webdriver function to retry
    :param args: arguments
    :return: Value, that is returned by a function
    :raises NoSuchElementException: if we could not find an element after retries
    :raises Exception: if func raises it
    """
    assert (retries > 0)
    value = None
    for i in range(retries, -1, -1):
        try:
            value = func(*args)
        except exc:
            if not i:
                raise
            continue
        except Exception:
            raise
        else:
            if value is not None:
                break
    return value


def get_currency(price_text):
    currency = price_text.split(' ')[-1].lower()

    if currency in ROUBLES:
        return 'RUB'

    unicode_currency = filter(None, map(parse_unicode_currency, price_text))
    iso_currency = filter(None, map(_iso_currency, grouped(price_text, 3)))
    try:
        return next(chain(unicode_currency, iso_currency))
    except StopIteration:
        return None


def _iso_currency(currency):
    currency = currency.upper()
    if currency in CURRENCIES:
        return currency
    else:
        return None


def parse_unicode_currency(currency):
    if currency == '$':
        return 'USD'

    if currency == '€':
        return 'EUR'

    if currency == '₽':
        return 'RUB'

    if currency == '£':
        return 'GBP'

    return None


def parse_price(price_text):
    subs = (
        (r'[^\d\.,]', ''),  # удаляем всё кроме цифр и разделителей
        (',', '.'),
        (r'\.*$', '')  # убираем оставшиеся на конце разделители
    )
    for pattern, repl in subs:
        price_text = re.sub(pattern, repl, price_text)
    return float(price_text)


class ErrorContainer(object):
    exc_info = None

    def was_caught(self):
        return bool(self.exc_info)

    def reraise(self):
        raise self.exc_info[0].with_traceback(*self.exc_info[1:])


@contextmanager
def catch_exc(*exc):
    # type: (*Type[Exception]) -> ContextManager[ErrorContainer]
    error = ErrorContainer()
    try:
        yield error
    except exc:
        error.exc_info = sys.exc_info()
