package main

import (
	"context"
	"fmt"
	"runtime"

	"github.com/jonboulle/clockwork"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/core/log/zap"
	"a.yandex-team.ru/library/go/maxprocs"
	"a.yandex-team.ru/travel/avia/library/go/searchcontext"
	"a.yandex-team.ru/travel/avia/search_results_queue_producer/internal/app"
	"a.yandex-team.ru/travel/avia/search_results_queue_producer/internal/dicts"
	logbrokersource "a.yandex-team.ru/travel/avia/search_results_queue_producer/internal/logbroker"
	"a.yandex-team.ru/travel/avia/search_results_queue_producer/internal/searchresultscache"
	"a.yandex-team.ru/travel/avia/search_results_queue_producer/internal/service/dynamicresources"
	"a.yandex-team.ru/travel/avia/search_results_queue_producer/internal/service/profiling"
	"a.yandex-team.ru/travel/avia/search_results_queue_producer/internal/service/qidconsumer"
	"a.yandex-team.ru/travel/avia/search_results_queue_producer/internal/service/qidparser"
	"a.yandex-team.ru/travel/avia/search_results_queue_producer/internal/service/queueproducer"
	"a.yandex-team.ru/travel/avia/search_results_queue_producer/internal/service/readiness"
	"a.yandex-team.ru/travel/avia/search_results_queue_producer/internal/service/shutdown"
	"a.yandex-team.ru/travel/library/go/configuration"
	"a.yandex-team.ru/travel/library/go/logbroker"
	"a.yandex-team.ru/travel/library/go/logging"
	"a.yandex-team.ru/travel/library/go/metrics"
	metricserver "a.yandex-team.ru/travel/library/go/metrics/server"
)

func main() {
	maxprocs.AdjustAuto()

	ctx, ctxCancel := context.WithCancel(context.Background())
	config := configuration.NewDefaultConfitaLoader()
	err := config.Load(ctx, &app.Cfg)

	if err != nil {
		fmt.Println("can not load configuration:", err)
		ctxCancel()
		return
	}

	logger, err := logging.NewDeploy(&app.Cfg.Logging)
	if err != nil {
		fmt.Println("failed to create logger, err:", err)
		ctxCancel()
		return
	}

	defer func() {
		err = logger.L.Sync()
		if err != nil {
			fmt.Println("failed to close logger:", err)
		}
		ctxCancel()
	}()

	logger.Infof("start with GOMAXPROCS = %d", runtime.GOMAXPROCS(0))

	rootRegistry := metrics.NewRegistryWithDeployTagsAndExplicitHost()
	// in order to make it possible to collect arbitrary application metrics through metrics.GlobalAppMetrics
	appMetrics := metrics.NewAppMetrics(rootRegistry.WithPrefix("app"))
	metrics.SetGlobalAppMetrics(appMetrics)
	metrics.RunPerfMetricsUpdater(rootRegistry, app.Cfg.Metrics.PerfMetricsRefreshInterval)

	dictsRegistry, err := dicts.NewRegistry(app.Cfg.Dicts, logger)
	if err != nil {
		logger.Error("failed to create dicts registry", log.Error(err))
		return
	}

	go func() {
		err = metricserver.RunMetricsHTTPServer(context.Background(), app.Cfg.Metrics, logger, rootRegistry)
		if err != nil {
			logger.Fatal("Error while starting metrics server", log.Error(err))
		}
	}()

	sessionPool, err := searchresultscache.GetSessionPool(ctx, app.Cfg.SearchResultsCache)
	if err != nil {
		logger.Fatal("failed to create session pool for SearchResultsCache", log.Error(err))
		return
	}
	searchResultsCache, err := searchresultscache.NewSearchResultsCache(logger, sessionPool, app.Cfg.SearchResultsCache)
	if err != nil {
		logger.Fatal("failed to create SearchResultsCache", log.Error(err))
		return
	}

	clock := clockwork.NewRealClock()
	var searchResultsProducer *queueproducer.Producer
	if app.Cfg.DryRun {
		nopProducer := queueproducer.NewNopProducer()
		searchResultsProducer = queueproducer.NewProducer(logger, nopProducer, searchResultsCache, clock)
	} else {
		logbrokerProducer, err := logbroker.NewProducer(
			app.Cfg.SearchResultsQueueProducer.Topic,
			app.Cfg.SearchResultsQueueProducer.Endpoint,
			logbrokersource.GetDeploySourceID(app.Cfg.SearchResultsQueueProducer.ProducerID),
			logbroker.NewOAuthCredentialsProvider(app.Cfg.SearchResultsQueueProducer.Token),
			logger,
			logbroker.WithoutSeqNo(),
		)
		if err != nil {
			logger.Fatal("failed to create logbroker producer", log.Error(err))
		}
		err = logbrokerProducer.Run(ctx)
		if err != nil {
			logger.Fatal("failed to run logbroker producer", log.Error(err))
		}
		defer logbrokerProducer.Close()
		searchResultsProducer = queueproducer.NewProducer(logger, logbrokerProducer, searchResultsCache, clock)
	}

	rawQIDsChan := make(chan string)
	if err != nil {
		logger.Fatal("failed to create qid consumer", log.Error(err))
		return
	}
	qidsChan := make(chan searchcontext.QID)
	qidParser := qidparser.NewParser(logger)
	qidParser.Parse(rawQIDsChan, qidsChan)
	qidConsumerService := qidconsumer.NewService(logger, app.Cfg.QIDConsumer)
	go qidConsumerService.Run(ctx, rawQIDsChan)
	go searchResultsProducer.ProduceFromQueryIDsChan(qidsChan)

	runProfilingService()

	waitShutdown := runShutdownServer(logger, ctxCancel)
	runReadinessServer(qidConsumerService)
	runDynamicResourcesService(logger, dictsRegistry)

	waitShutdown()
	logger.Info("the instance has been shut down")
}

func runProfilingService() {
	profilingService := profiling.NewService(app.Cfg.Profiling)
	profilingService.BackroundRun()
}

func runShutdownServer(
	logger *zap.Logger,
	ctxCancel context.CancelFunc,
) (waitShutdown func()) {
	shutdownService := shutdown.NewService(
		app.Cfg.Shutdown,
		func() {
			logger.Info("the instance is going to be shut down")
			ctxCancel()
		},
	)
	return shutdownService.BackroundRun()
}

func runReadinessServer(qidConsumerService *qidconsumer.Service) {
	readinessService := readiness.NewService(
		app.Cfg.Readiness,
		func() bool {
			return qidConsumerService.Ready()
		},
	)
	readinessService.BackroundRun()
}

func runDynamicResourcesService(logger *zap.Logger, dictsRegistry *dicts.Registry) {
	dynamicResourceService := dynamicresources.NewService(
		logger,
		app.Cfg.DynamicResources,
		dictsRegistry.OnUpdate,
	)
	dynamicResourceService.BackroundRun()
}
