package dicts

import (
	"fmt"
	"io"
	"path/filepath"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/travel/library/go/dicts/base"
)

type Config struct {
	ResourcesPath       string `config:"DICTS_RESOURCES_PATH" yaml:"resources_path"`
	UseDynamicResources bool   `yaml:"use_dynamic_resources"`
}

var DefaultConfig = Config{
	ResourcesPath:       "/app/data",
	UseDynamicResources: false,
}

type Registry struct {
	writerByPath         map[string]UpdatableWriter
	settlements          *SettlementsRepository
	stations             *StationsRepository
	stationToSettlements *StationToSettlementsRepository
	stationCodes         *StationCodesRepository
	timeZones            *TimeZonesRepository
	config               Config
}

type RepositoryUpdater interface {
	Populate(writer io.Writer) error
}

type UpdatableWriter interface {
	io.Writer
	UpdateFromSource(RepositoryUpdater) error
}

func populateMany(path2writer map[string]UpdatableWriter) error {
	for path, writer := range path2writer {
		bytesIterator, err := base.BuildIteratorFromFile(path)
		if err != nil {
			return fmt.Errorf("pkg.dicts.registry:populateMany %w", err)
		}
		err = bytesIterator.Populate(writer)
		if err != nil {
			return fmt.Errorf("pkg.dicts.registry:populateMany %w", err)
		}
	}
	return nil
}

func NewRegistry(c Config, logger log.Logger) (*Registry, error) {
	r := &Registry{
		settlements:          NewSettlementsRepository(),
		stations:             NewStationsRepository(),
		stationToSettlements: NewStationToSettlementsRepository(),
		stationCodes:         NewStationCodesRepository(),
		timeZones:            NewTimeZonesRepository(logger),
		config:               c,
	}

	resourcePath, err := getResourcePath(c)
	if err != nil {
		return nil, fmt.Errorf("pkg.dicts.registry:NewRegistry %w", err)
	}

	r.updateWriterByPath(resourcePath)
	err = populateMany(r.writerByPath)
	if err != nil {
		return nil, fmt.Errorf("pkg.dicts.registry:NewRegistry %w", err)
	}
	return r, nil
}

func (r *Registry) GetSettlementsRepository() *SettlementsRepository {
	return r.settlements
}

func (r *Registry) GetStationsRepository() *StationsRepository {
	return r.stations
}

func (r *Registry) GetStationToSettlementRepository() *StationToSettlementsRepository {
	return r.stationToSettlements
}

func (r *Registry) GetStationCodesRepository() *StationCodesRepository {
	return r.stationCodes
}

func (r *Registry) GetTimeZonesRepository() *TimeZonesRepository {
	return r.timeZones
}

func (r *Registry) OnUpdate() error {
	resourcePath, err := getResourcePath(r.config)
	if err != nil {
		return fmt.Errorf("pkg.dicts.registry:OnUpdate %w", err)
	}
	r.updateWriterByPath(resourcePath)

	for path, writer := range r.writerByPath {
		bytesIterator, err := base.BuildIteratorFromFile(path)
		if err != nil {
			return fmt.Errorf("pkg.dicts.registry:OnUpdate %w", err)
		}
		if err := writer.UpdateFromSource(bytesIterator); err != nil {
			return fmt.Errorf("pkg.dicts.registry:OnUpdate %w", err)
		}
	}
	return nil
}

func (r *Registry) updateWriterByPath(resourcePath string) {
	r.writerByPath = map[string]UpdatableWriter{
		resourcePath + "/settlement.data":         r.settlements,
		resourcePath + "/station.data":            r.stations,
		resourcePath + "/settlement2station.data": r.stationToSettlements,
		resourcePath + "/station_code.data":       r.stationCodes,
		resourcePath + "/timezone.data":           r.timeZones,
	}
}

func getResourcePath(config Config) (string, error) {
	resourcePath := config.ResourcesPath
	if config.UseDynamicResources {
		symlinkTarget, err := filepath.EvalSymlinks(config.ResourcesPath)
		if err != nil {
			return "", err
		}
		resourcePath = filepath.Dir(symlinkTarget)
	}
	return resourcePath, nil
}
