package protobuilder

import (
	"strconv"
	"strings"
	"sync"
	"time"

	"github.com/jonboulle/clockwork"

	"a.yandex-team.ru/travel/avia/library/go/searchcontext"
	"a.yandex-team.ru/travel/avia/library/proto/common/v1"
	priceprediction "a.yandex-team.ru/travel/avia/library/proto/price_prediction/v1"
	resultproto "a.yandex-team.ru/travel/avia/library/proto/search_result/v1"
	"a.yandex-team.ru/travel/avia/search_results_queue_producer/internal/searchresultscache"
	travelproto "a.yandex-team.ru/travel/proto"
)

func Build(queryID searchcontext.QID, results []*searchresultscache.SearchResult, clock clockwork.Clock) *resultproto.Result {
	result := &resultproto.Result{
		Qid:          queryID.QID,
		PointFrom:    mapPoint(queryID.QKey.PointFromKey),
		PointTo:      mapPoint(queryID.QKey.PointToKey),
		DateForward:  mapDate(queryID.QKey.DateForward),
		DateBackward: nil,
		ServiceClass: mapServiceClass(queryID.QKey.Class),
		Passengers: &common.Passengers{
			Adults:   uint32(queryID.QKey.Adults),
			Children: uint32(queryID.QKey.Children),
			Infants:  uint32(queryID.QKey.Infants),
		},
		NationalVersion: mapNationalVersion(queryID.QKey.NationalVersion),
		Flights:         mapFlights(results),
		Variants:        mapVariants(results, clock),
	}
	if !queryID.QKey.DateBackward.IsZero() {
		result.DateBackward = mapDate(queryID.QKey.DateBackward)
	}
	return result
}

func mapVariants(results []*searchresultscache.SearchResult, clock clockwork.Clock) []*resultproto.Variant {
	totalVariants := 0
	for _, result := range results {
		totalVariants += len(result.Variants.Value.Fares)
	}
	variants := make([]*resultproto.Variant, 0, totalVariants)
	for _, result := range results {
		if result.Meta.Value.Expire <= clock.Now().UTC().Unix() {
			continue
		}
		for _, fare := range result.Variants.Value.Fares {
			variants = append(variants, mapVariant(result, fare))
		}
	}
	return variants
}

func mapVariant(searchResult *searchresultscache.SearchResult, fare searchresultscache.Fare) *resultproto.Variant {
	charter := false
	if fare.Charter != nil {
		charter = *fare.Charter
	}
	return &resultproto.Variant{
		PartnerCode: searchResult.PartnerCode,
		Charter:     charter,
		SelfConnect: fare.Selfconnect,
		Forward:     mapSegments(fare.Route[0], fare.Baggage[0], fare.FareCodes[0]),
		Backward:    mapSegments(fare.Route[1], fare.Baggage[1], fare.FareCodes[1]),
		CreatedAt:   int64(searchResult.CreatedAt),
		ExpiredAt:   int64(searchResult.ExpiresAt),
		Price: &common.Price{
			Currency: fare.Tariff.Currency,
			Value:    fare.Tariff.Value,
		},
		PricePredictionResult: mapPricePredictionResult(fare.PriceCategory),
	}
}

func mapPricePredictionResult(rawPredictionResult string) priceprediction.PricePredictionResult {
	switch rawPredictionResult {
	case "good":
		return priceprediction.PricePredictionResult_PRICE_PREDICTION_RESULT_GOOD
	case "bad":
		return priceprediction.PricePredictionResult_PRICE_PREDICTION_RESULT_BAD
	case "unknown":
		fallthrough
	default:
		return priceprediction.PricePredictionResult_PRICE_PREDICTION_RESULT_UNKNOWN
	}
}

func mapSegments(rawSegments []string, rawBaggageCodes []string, fareCodes []string) []*resultproto.FlightSegment {
	segments := make([]*resultproto.FlightSegment, 0, len(rawSegments))
	for i, s := range rawSegments {
		segments = append(segments, mapSegment(s, rawBaggageCodes[i], fareCodes[i]))
	}
	return segments
}

func mapSegment(flightKey, baggageKey, fareCode string) *resultproto.FlightSegment {
	return &resultproto.FlightSegment{
		FlightKey:  flightKey,
		FareCode:   fareCode,
		FareFamily: "", // TODO: fill fare_families in ticket-daemon
		Baggage:    mapBaggage(baggageKey),
	}
}

var baggageCodeCache = sync.Map{}

func mapBaggage(baggageCode string) *common.Baggage {
	if checkedBaggage, ok := baggageCodeCache.Load(baggageCode); ok {
		return checkedBaggage.(*common.Baggage)
	}
	if baggageCode == "" {
		baggage := &common.Baggage{Included: false}
		baggageCodeCache.Store(baggageCode, baggage)
		return baggage
	}
	processedCode := strings.ReplaceAll(baggageCode, "N", "p")
	processedCode = strings.ReplaceAll(processedCode, "d", "p")
	parts := strings.Split(processedCode, "p")
	var included, pieces, weight int
	if parsed, err := strconv.Atoi(parts[0]); err == nil {
		included = parsed
	}
	if parsed, err := strconv.Atoi(parts[1]); err == nil {
		pieces = parsed
	}
	if parsed, err := strconv.Atoi(parts[2]); err == nil {
		weight = parsed
	}
	if included == 0 {
		baggage := &common.Baggage{Included: false}
		baggageCodeCache.Store(baggageCode, baggage)
		return baggage
	}
	baggage := &common.Baggage{
		Included: true,
		Pieces:   uint32(pieces),
		Weight:   uint32(weight),
	}
	baggageCodeCache.Store(baggageCode, baggage)
	return baggage

}

func mapFlights(results []*searchresultscache.SearchResult) map[string]*resultproto.Flight {
	flights := make(map[string]*resultproto.Flight)
	for _, result := range results {
		for flightKey, flight := range result.Variants.Value.Flights {
			flights[flightKey] = mapFlight(flightKey, flight)
		}
	}
	return flights
}

func mapFlight(flightKey string, flight *searchresultscache.Flight) *resultproto.Flight {
	return &resultproto.Flight{
		Key:            flightKey,
		Number:         flight.Number,
		CompanyId:      uint64(flight.Company),
		StationFromId:  uint64(flight.From),
		StationToId:    uint64(flight.To),
		LocalArrival:   flight.Arrival.Local,
		LocalDeparture: flight.Departure.Local,
		UtcArrival:     convertToUTC(flight.Arrival.Local, flight.Arrival.Offset),
		UtcDeparture:   convertToUTC(flight.Departure.Local, flight.Departure.Offset),
	}
}

func convertToUTC(localTimeRaw string, offsetMinutes int) string {
	const layout = "2006-01-02T15:04:05"
	localTime, _ := time.Parse(layout, localTimeRaw)
	return localTime.Add(-time.Duration(offsetMinutes) * time.Minute).Format(layout)
}

var stringToNationalVersion = map[string]common.NationalVersion{
	"ru":  common.NationalVersion_NATIONAL_VERSION_RU,
	"com": common.NationalVersion_NATIONAL_VERSION_COM,
	"tr":  common.NationalVersion_NATIONAL_VERSION_TR,
	"kz":  common.NationalVersion_NATIONAL_VERSION_KZ,
	"ua":  common.NationalVersion_NATIONAL_VERSION_UA,
}

func mapNationalVersion(rawNationalVersion string) common.NationalVersion {
	if nationalVersion, ok := stringToNationalVersion[rawNationalVersion]; ok {
		return nationalVersion
	}
	return common.NationalVersion_NATIONAL_VERSION_RU
}

func mapServiceClass(serviceClass string) common.ServiceClass {
	if serviceClass == "economy" {
		return common.ServiceClass_SERVICE_CLASS_ECONOMY
	}
	return common.ServiceClass_SERVICE_CLASS_BUSINESS
}

func mapDate(date time.Time) *travelproto.TDate {
	return &travelproto.TDate{
		Year:  int32(date.Year()),
		Month: int32(date.Month()),
		Day:   int32(date.Day()),
	}
}

var charToPointType = map[string]common.PointType{
	"c": common.PointType_POINT_TYPE_SETTLEMENT,
	"s": common.PointType_POINT_TYPE_STATION,
	"r": common.PointType_POINT_TYPE_REGION,
	"l": common.PointType_POINT_TYPE_COUNTRY,
}

func mapPoint(pointKey string) *common.Point {
	id, _ := strconv.Atoi(pointKey[1:])
	pointTypeChar := string(pointKey[0])
	return &common.Point{
		Type: charToPointType[pointTypeChar],
		Id:   uint64(id),
	}
}
