package searchresultscache

import (
	"bytes"
	"compress/zlib"
	"context"
	"encoding/json"
	"io/ioutil"
	"time"

	"github.com/opentracing/opentracing-go"

	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/metrics"
)

type VariantsScanner struct {
	Context context.Context
	Value   Variants
	Error   error
}

func (s VariantsScanner) Get() (string, bool) {
	panic("conversion from Variants to json is not implemented")
}

func (s *VariantsScanner) Set(v string) {
	span, _ := opentracing.StartSpanFromContext(s.Context, "Deserializing Variants")
	now := time.Now()
	defer func() {
		metrics.GlobalWizardMetrics().FaresProtobufDeserialization.RecordDuration(time.Since(now))
	}()
	defer span.Finish()
	var variants Variants
	zlibReader, err := zlib.NewReader(bytes.NewReader([]byte(v)))
	var jsonBytes []byte
	if err != nil {
		if err == zlib.ErrHeader || err == zlib.ErrChecksum {
			jsonBytes = []byte(v)
		} else {
			s.Error = err
			return
		}
	} else if jsonBytes, err = ioutil.ReadAll(zlibReader); err != nil {
		s.Error = err
		return
	}
	err = json.Unmarshal(jsonBytes, &variants)
	if err != nil {
		s.Error = err
		return
	}
	*s = VariantsScanner{Value: variants}
}

type MetaScanner struct {
	Context context.Context
	Value   Meta
	Error   error
}

func (s MetaScanner) Get() (string, bool) {
	panic("conversion from Meta to json is not implemented")
}

func (s *MetaScanner) Set(v string) {
	span, _ := opentracing.StartSpanFromContext(s.Context, "Deserializing Meta")
	defer span.Finish()
	meta := Meta{}
	if err := json.Unmarshal([]byte(v), &meta); err != nil {
		s.Error = err
		return
	}
	*s = MetaScanner{Value: meta}
}

//go:generate ydbgen -wrap optional

//ydb:gen
//ydb:set wrap:none
type SearchResult struct {
	PointFrom       string          `ydb:"type:utf8?"`
	PointTo         string          `ydb:"type:utf8?"`
	DateForward     uint32          `ydb:"type:uint32?"`
	DateBackward    uint32          `ydb:"type:uint32?"`
	Class           uint8           `ydb:"column:klass,type:uint8?"`
	Passengers      uint32          `ydb:"type:uint32?"`
	NationalVersion string          `ydb:"type:utf8?"`
	Lang            string          `ydb:"type:utf8?"`
	PartnerCode     string          `ydb:"type:utf8?"`
	Variants        VariantsScanner `ydb:"type:string?"`
	Meta            MetaScanner     `ydb:"type:string?"`
	CreatedAt       uint64          `ydb:"type:uint64?"`
	ExpiresAt       uint64          `ydb:"type:uint64?"`
}

func NewSearchResult(ctx context.Context) *SearchResult {
	return &SearchResult{Variants: VariantsScanner{Context: ctx}, Meta: MetaScanner{Context: ctx}}
}
