package qidconsumer

import (
	"bytes"
	"context"
	"encoding/json"
	"time"

	"github.com/cenkalti/backoff/v4"

	"a.yandex-team.ru/kikimr/public/sdk/go/persqueue"
	"a.yandex-team.ru/kikimr/public/sdk/go/persqueue/log/corelogadapter"
	"a.yandex-team.ru/kikimr/public/sdk/go/ydb"
	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/travel/avia/library/go/logbroker"
	"a.yandex-team.ru/travel/library/go/metrics"
)

type QIDConsumer struct {
	baseReader  *logbroker.Reader
	logger      log.Logger
	rawQidsChan chan<- string
}

func NewQIDConsumer(
	ctx context.Context,
	config Config,
	endpoint string,
	rawQIDsChan chan<- string,
	logger log.Logger,
) (*QIDConsumer, error) {
	readerOptions := persqueue.ReaderOptions{
		Endpoint:              endpoint,
		Credentials:           ydb.AuthTokenCredentials{AuthToken: config.Token},
		Consumer:              config.ProcessedQIDTopic.Consumer,
		Logger:                corelogadapter.New(logger),
		Topics:                []persqueue.TopicInfo{{Topic: config.ProcessedQIDTopic.Topic}},
		MaxReadSize:           config.ProcessedQIDTopic.MaxReadSize,
		MaxReadMessagesCount:  config.ProcessedQIDTopic.MaxReadMessageCount,
		DecompressionDisabled: false,
		RetryOnFailure:        true,
	}

	var err error
	qidConsumer := &QIDConsumer{logger: logger, rawQidsChan: rawQIDsChan}
	qidConsumer.baseReader, err = logbroker.NewReader(
		ctx,
		readerOptions,
		logbroker.WithOnBatchReceived(qidConsumer.onBatchReceived),
		logbroker.WithOnMessage(qidConsumer.onMessage),
	)
	if err != nil {
		return nil, err
	}
	return qidConsumer, nil
}

func (c *QIDConsumer) onBatchReceived(batch persqueue.MessageBatch) {
	c.logger.Debug(
		"Received batch",
		log.Any("topic", batch.Topic),
		log.Any("partition", batch.Partition),
		log.Any("messages", len(batch.Messages)),
	)
}

func (c *QIDConsumer) onMessage(message persqueue.ReadMessage) {
	c.logger.Debug(
		"Received QID message",
		log.Any("seqNo", message.SeqNo),
		log.Any("source-id", string(message.SourceID)),
		log.Any("created", message.CreateTime),
		log.Any("written", message.WriteTime),
		log.Any("codec", message.Codec),
	)
	for _, data := range bytes.Split(message.Data, []byte("\n")) {
		if len(data) == 0 {
			continue
		}
		countMessageMetric("total")
		qidMessage := QIDMessage{}
		err := json.Unmarshal(data, &qidMessage)
		if err != nil {
			c.logger.Error("invalid QIDMessage", log.ByteString("data", data))
			countMessageMetric("invalid")
		} else {
			c.rawQidsChan <- qidMessage.QID
			countMessageMetric("correct")
		}
	}
}

func countMessageMetric(metricType string) {
	metrics.GlobalAppMetrics().GetOrCreateCounter(metricsPrefix, map[string]string{"type": metricType}, receievedMetricName).Inc()
}

func (c *QIDConsumer) Read() error {
	return backoff.RetryNotify(
		c.baseReader.Read,
		backoff.WithMaxRetries(backoff.NewConstantBackOff(10*time.Second), 3),
		func(err error, _ time.Duration) {
			c.logger.Error("failed to read", log.Error(err))
		},
	)
}
