package qidconsumer

import (
	"context"
	"fmt"
	"time"

	"github.com/cenkalti/backoff/v4"
	"go.uber.org/atomic"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/travel/library/go/metrics"
)

type Service struct {
	logger          log.Logger
	config          Config
	activeConsumers atomic.Int32
}

func NewService(logger log.Logger, config Config) *Service {
	return &Service{logger: logger.WithName("QIDConsumerService"), config: config}
}

func (s *Service) Ready() bool {
	return s.config.MinActiveConsumers <= s.activeConsumers.Load()
}

func (s *Service) Run(ctx context.Context, rawQIDsChan chan string) {
	if s.config.Disabled {
		return
	}
	for _, endpoint := range s.config.ReadEndpoints {
		go s.runSingleEndpoint(ctx, endpoint, rawQIDsChan)
	}
}

func (s *Service) runSingleEndpoint(ctx context.Context, endpoint string, rawQIDsChan chan string) {
	isStopped := false
	_ = backoff.RetryNotify(
		func() error {
			select {
			case <-ctx.Done():
				isStopped = true
				return backoff.Permanent(fmt.Errorf("consumer has been stopped"))
			default:
			}
			qidConsumer, err := NewQIDConsumer(ctx, s.config, endpoint, rawQIDsChan, s.logger.WithName("QIDConsumer-"+endpoint))
			if err != nil {
				return fmt.Errorf("failed to start qid consumer: %w", err)
			}
			metrics.GlobalAppMetrics().GetOrCreateGauge(metricsPrefix, map[string]string{"endpoint": endpoint}, aliveConsumers).Set(1)
			defer metrics.GlobalAppMetrics().GetOrCreateGauge(metricsPrefix, map[string]string{"endpoint": endpoint}, aliveConsumers).Set(0)
			s.activeConsumers.Inc()
			err = qidConsumer.Read()
			s.activeConsumers.Dec()
			if err != nil {
				s.logger.Error("qidConsumer has been closed unexpectedly", log.Error(err))
				return err
			}
			return nil
		},
		backoff.NewConstantBackOff(10*time.Second),
		func(err error, duration time.Duration) {
			if isStopped {
				return
			}
			s.logger.Error("failed to run single endpoint consumer. The operation will be retried", log.Error(err))
		},
	)
}
