package queueproducer

import (
	"context"
	"strconv"
	"sync"
	"time"

	timeformats "cuelang.org/go/pkg/time"
	"github.com/golang/protobuf/proto"
	"github.com/jonboulle/clockwork"

	"a.yandex-team.ru/kikimr/public/sdk/go/persqueue"
	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/core/log/ctxlog"
	"a.yandex-team.ru/travel/avia/library/go/searchcontext"
	"a.yandex-team.ru/travel/avia/library/proto/common/v1"
	"a.yandex-team.ru/travel/avia/search_results_queue_producer/internal/protobuilder"
	"a.yandex-team.ru/travel/avia/search_results_queue_producer/internal/searchresultscache"
	"a.yandex-team.ru/travel/library/go/metrics"
	"a.yandex-team.ru/travel/proto"
)

type Config struct {
	Topic      string `config:"SEARCH_RESULTS_TOPIC" yaml:"topic"`
	ProducerID string `config:"SEARCH_RESULTS_PRODUCER_ID" yaml:"producer_id"`
	Endpoint   string `config:"LOGBROKER_PRODUCER_ENDPOINT" yaml:"endpoint"`
	Token      string `config:"LOGBROKER_PRODUCER_TOKEN,required" yaml:"token"`
}

var DefaultConfig = Config{
	Topic:      "/avia/development/miralx/search-results-queue",
	Endpoint:   "logbroker.yandex.net",
	ProducerID: "search-results-producer",
}

type LogbrokerProducer interface {
	Write(msg proto.Message) error
	Stat() persqueue.WriterStat
}

type NopProducer struct {
}

func NewNopProducer() *NopProducer {
	return &NopProducer{}
}

func (n *NopProducer) Write(proto.Message) error {
	return nil
}

func (n *NopProducer) Stat() persqueue.WriterStat {
	return persqueue.WriterStat{}
}

type ResultsCache interface {
	GetSearchResultsByQueryKey(ctx context.Context, queryKey searchcontext.QKey) ([]*searchresultscache.SearchResult, error)
}

type Producer struct {
	logger log.Logger

	logbrokerProducer LogbrokerProducer
	resultsCache      ResultsCache
	clock             clockwork.Clock
}

func NewProducer(logger log.Logger, logbrokerProducer LogbrokerProducer, resultsCache ResultsCache, clock clockwork.Clock) *Producer {
	go sendProducerStat(logbrokerProducer)
	return &Producer{logger: logger, logbrokerProducer: logbrokerProducer, resultsCache: resultsCache, clock: clock}
}

func sendProducerStat(producer LogbrokerProducer) {
	t := time.NewTicker(5 * time.Second)
	defer t.Stop()
	for range t.C {
		stat := producer.Stat()
		metrics.GlobalAppMetrics().GetOrCreateGauge(metricsPrefix, nil, "inflight").Set(float64(stat.Inflight))
		metrics.GlobalAppMetrics().GetOrCreateGauge(metricsPrefix, nil, "mem_usage").Set(float64(stat.MemUsage))
	}
}

func (p *Producer) ProduceFromQueryIDsChan(queryIDsChan <-chan searchcontext.QID) {
	counter := 0
	const waitGroupSize = 100
	wg := sync.WaitGroup{}
	wg.Add(waitGroupSize)
	for queryID := range queryIDsChan {
		go p.processQueryID(queryID, &wg)
		counter += 1
		if counter == 100 {
			wg.Wait()
			counter = 0
			wg = sync.WaitGroup{}
			wg.Add(waitGroupSize)
		}
	}
}

func (p *Producer) processQueryID(queryID searchcontext.QID, wg *sync.WaitGroup) {
	defer wg.Done()
	ctx := context.Background()
	ctx = ctxlog.WithFields(ctx, log.String("queryID", queryID.QID))
	p.logger.Info("start processing qid", ctxlog.ContextFields(ctx)...)
	searchResults, err := p.resultsCache.GetSearchResultsByQueryKey(ctx, queryID.QKey)
	if err != nil {
		sendSearchResultsMetric(map[string]string{"status": "error", "phase": "get_from_cache"})
		p.logger.Error("failed to get search results for queryID", append(ctxlog.ContextFields(ctx), log.Error(err))...)
		return
	}
	searchResultsProto := protobuilder.Build(queryID, searchResults, p.clock)
	err = p.logbrokerProducer.Write(searchResultsProto)
	if err != nil {
		sendSearchResultsMetric(map[string]string{"status": "error", "phase": "write_to_logbroker"})
		p.logger.Error("failed to write proto-message into logbroker", append(ctxlog.ContextFields(ctx), log.Error(err))...)
	} else {
		sendSearchResultsMetric(map[string]string{"status": "success"})
		ctxlog.Info(
			ctx,
			p.logger,
			"search result has been sent to logbroker successfully",
			log.Int("variantsCount", len(searchResultsProto.Variants)),
			log.Int("flightsCount", len(searchResultsProto.Flights)),
			log.String("pointFrom", pointToPointKey(searchResultsProto.PointFrom)),
			log.String("pointTo", pointToPointKey(searchResultsProto.PointTo)),
			log.String("dateForward", dateProtoToString(searchResultsProto.DateForward)),
			log.String("dateBackward", dateProtoToString(searchResultsProto.DateBackward)),
		)
	}
}

func dateProtoToString(dateProto *travel_commons_proto.TDate) string {
	if dateProto == nil || dateProto.Year == 0 {
		return ""
	}
	date := time.Date(int(dateProto.Year), time.Month(dateProto.Month), int(dateProto.Day), 0, 0, 0, 0, time.UTC)
	return date.Format(timeformats.RFC3339Date)
}

func pointToPointKey(point *common.Point) string {
	pointKey := ""
	switch point.Type {
	case common.PointType_POINT_TYPE_STATION:
		pointKey += "s"
	case common.PointType_POINT_TYPE_SETTLEMENT:
		pointKey += "c"
	case common.PointType_POINT_TYPE_REGION:
		pointKey += "r"
	case common.PointType_POINT_TYPE_COUNTRY:
		pointKey += "l"
	}

	return pointKey + strconv.Itoa(int(point.Id))
}

func sendSearchResultsMetric(tags map[string]string) {
	metrics.GlobalAppMetrics().GetOrCreateCounter(metricsPrefix, tags, searchResultsMetricName).Inc()
}
