package shutdown

import (
	"context"
	"encoding/json"
	"net/http"
	"sync"

	"github.com/go-chi/chi/v5"

	"a.yandex-team.ru/library/go/httputil/headers"
)

type Config struct {
	Addr  string
	Route string
}

var DefaultConfig = Config{
	Addr:  "[::]:9002",
	Route: "/shutdown",
}

type Service struct {
	onShutdown func()
	config     Config
	shutdown   chan bool
}

func NewService(config Config, onShutdown func()) *Service {
	return &Service{config: config, onShutdown: onShutdown, shutdown: make(chan bool, 1)}
}

func (s *Service) BackroundRun() (waitShutdown func()) {
	router := chi.NewRouter()
	router.Get(s.config.Route, s.onShutdownHandler)
	server := http.Server{Addr: s.config.Addr, Handler: router}
	wg := sync.WaitGroup{}
	wg.Add(1)
	go func() {
		_ = server.ListenAndServe()
	}()
	go func() {
		<-s.shutdown
		_ = server.Shutdown(context.Background())
		wg.Done()
	}()
	return wg.Wait
}

type shutdownResponse struct {
	Status string `json:"status"`
}

var okResponse, _ = json.Marshal(shutdownResponse{"OK"})

func (s *Service) onShutdownHandler(w http.ResponseWriter, r *http.Request) {
	s.onShutdown()
	w.WriteHeader(http.StatusOK)
	w.Header().Set(headers.ContentTypeKey, headers.TypeApplicationJSON.String())
	_, _ = w.Write(okResponse)
	s.shutdown <- true
}
