# coding=utf-8
from __future__ import unicode_literals

import logging
import re
import psycopg2
import traceback
from collections import defaultdict
from datetime import datetime, timedelta
from typing import List
from werkzeug.utils import secure_filename

from travel.avia.shared_flights.lib.python.consts.consts import MIN_APM_FLIGHT_BASE_ID, MIN_APM_FLIGHT_PATTERN_ID
from travel.avia.shared_flights.lib.python.db_models.apm_imported_file import ApmImportedFile
from travel.avia.shared_flights.lib.python.db_models.flight_base import ApmFlightBase
from travel.avia.shared_flights.lib.python.db_models.flight_pattern import ApmFlightPattern
from travel.avia.shared_flights.lib.python.date_utils.date_index import DateIndex
from travel.avia.shared_flights.lib.python.date_utils.date_mask import DateMaskMatcher
from travel.avia.shared_flights.lib.python.date_utils.date_matcher import DateMatcher
from travel.library.python.safexml.safe_xml_parser import safe_xml_fromstring
from travel.proto.dicts.rasp.carrier_pb2 import TCarrier
from travel.proto.shared_flights.snapshots.station_with_codes_pb2 import TStationWithCodes


logger = logging.getLogger(__name__)
YYYYMMDD = '%Y-%m-%d'


class Stoppoint(object):

    def __init__(self, station_code, station_iata, departure_time_int, arrival_time_int):
        self.station_code = station_code
        self.station_iata = station_iata
        self.departure_time_int = departure_time_int
        self.arrival_time_int = arrival_time_int


class ApmParser(object):

    def __init__(self):
        self._start_date = datetime.now() - timedelta(days=75)
        self._max_days = 440
        self._date_index = DateIndex(self._start_date)
        self._date_matcher = DateMatcher(self._date_index)
        self._min_flight_base_id = MIN_APM_FLIGHT_BASE_ID
        self._min_flight_pattern_id = MIN_APM_FLIGHT_PATTERN_ID

    def parse(self, apm_file, config, memo):
        apm_filename = 'unknown'
        error = ''
        messages = []
        if not memo:
            memo = ''
        response_data = {}
        if apm_file:
            text = apm_file.read()
            file_size = len(text)
            apm_filename = secure_filename(apm_file.filename)
            try:
                dmm = DateMaskMatcher(self._date_index, self._start_date, self._max_days)
                new_flight_bases = []
                new_flight_patterns = []

                db_conn = psycopg2.connect(self.get_psycopg2_conn_string(config))
                cursor = db_conn.cursor()
                stations = self.fetch_stations_to_dict(cursor)
                carriers = self.fetch_carriers_to_dict(cursor)

                flight_patterns_map, flight_patterns_masks = self.fetch_flight_patterns(cursor, dmm)

                root_el = safe_xml_fromstring((text))
                flights = []
                # fetch_flight_base_ids_to_list properly sets self.__min_flight_base_id
                existing_flight_base_ids = self.fetch_flight_base_ids_to_list(cursor)
                for channel_index, channel_el in enumerate(root_el.xpath('//channel'), 1):
                    t_type = channel_el.get('t_type', '').strip()
                    if t_type != 'plane':
                        messages.append('Warning: channel t_type is not plane ({}) for channel {}'.format(t_type, channel_index))
                        continue
                    group_el = channel_el.find('.//group')
                    if group_el is None:
                        messages.append('Warning: channel {} does not have any groups'.format(channel_index))
                        continue

                    stations_el = group_el.find('.//stations')
                    if stations_el is None:
                        messages.append('Warning: channel {} does not have any stations'.format(channel_index))
                        continue
                    for station_el in stations_el.xpath('.//station'):
                        code = station_el.get('code', '').strip()
                        if code not in stations:
                            upload_error = True
                            messages.append('Error: Unknown station code: {}'.format(code))

                    threads_el = group_el.find('.//threads')
                    if threads_el is None:
                        messages.append('Warning: channel {} does not have any threads'.format(channel_index))
                        continue
                    upload_error = False
                    for thread_el in threads_el.xpath('.//thread'):
                        cur_flight_bases = []
                        flight_number = thread_el.get('number', '').strip()
                        flight_title = thread_el.get('title', '').strip()
                        if not flight_number:
                            upload_error = True
                            messages.append('Flight with no number: {}'.format(flight_title))
                        # здесь flight_number - это "ДЕ 123", если он будет меньше 3 в длину,
                        # выражения flight_number[:2] и flight_number[2:] ниже будут вылетать по эксепшену
                        if len(flight_number) < 3:
                            upload_error = True
                            messages.append('Flight with invalid number: {} {}'.format(flight_number, flight_title))
                        flights.append(flight_number)

                        carrier, carrier_error = self.get_carrier(thread_el, carriers, messages, flight_number)
                        if carrier_error:
                            upload_error = True
                        messages.append('flight {} {}'.format(flight_number, flight_title))

                        stoppoints_el = thread_el.find('.//stoppoints')
                        if stoppoints_el is None:
                            messages.append('Warning: flight {} {} does not have any stop points'.format(flight_number, flight_title))
                            continue
                        stoppoints, stoppoints_error = self.get_stoppoints(stoppoints_el, flight_number, flight_title, stations, messages)
                        if stoppoints_error:
                            upload_error = True

                        schedules_el = thread_el.find('.//schedules')
                        if schedules_el is None:
                            messages.append('Warning: flight {} {} does not have any schedules'.format(flight_number, flight_title))
                            continue

                        if upload_error:
                            continue

                        for segment_number in range(1, len(stoppoints)):
                            stoppoint_from = stoppoints[segment_number-1]
                            stoppoint_to = stoppoints[segment_number]

                            fb = self.create_apm_flight_base(carrier, flight_number, segment_number, stoppoint_from, stoppoint_to)
                            errors = self.verify_flight_base(fb, flight_number, segment_number)
                            if errors:
                                upload_error = True
                                messages.extend(errors)

                            new_flight_bases.append(fb)
                            cur_flight_bases.append(fb)

                        if upload_error:
                            continue

                        date_mask = dmm.new_date_mask()
                        date_mask_to_remove = dmm.new_date_mask()
                        for schedule_el in schedules_el.xpath('.//schedule'):
                            self.process_schedule_elem(schedule_el, dmm, date_mask, date_mask_to_remove, flight_number, flight_title)

                        # remove new dates from existing flight patterns
                        flight_number_stripped = flight_number[2:].strip()
                        flight_key = self.get_flight_key(carrier.Id, flight_number_stripped)
                        for flight_pattern in flight_patterns_map.get(flight_key, []):
                            fp_mask = flight_patterns_masks[flight_pattern.id]
                            dmm.remove_mask(fp_mask, date_mask_to_remove)

                        for mask in dmm.generate_masks(date_mask):
                            for fb in cur_flight_bases:
                                fp = ApmFlightPattern()
                                fp.id = self._min_flight_pattern_id
                                self._min_flight_pattern_id += 1
                                fp.flight_base_id = fb.id
                                fp.marketing_carrier = carrier.Id
                                fp.marketing_carrier_iata = carrier.Iata if carrier.Iata else flight_number[:2]
                                fp.marketing_flight_number = flight_number_stripped
                                fp.bucket_key = fb.bucket_key
                                fp.flight_leg_key = '{}.{}'.format(fb.bucket_key, fb.itinerary_variation)
                                fp.operating_from = mask[0].replace('.', '-')
                                fp.operating_until = mask[1].replace('.', '-')
                                fp.operating_on_days = mask[2]
                                fp.leg_seq_number = fb.leg_seq_number
                                fp.is_administrative = False
                                fp.is_codeshare = False
                                fp.performance = 0
                                fp.arrival_day_shift = 0
                                fp.operating_flight_pattern_id = 0
                                new_flight_patterns.append(fp)

                if upload_error:
                    raise Exception('There were errors during upload, please see messages below for details.')

                # find out which old flight patterns / flight bases are no longer needed
                flight_bases_to_keep = set()
                deleted_flight_patterns_count = 0
                updated_flight_patterns = []
                for flight_patterns_list in flight_patterns_map.values():
                    for flight_pattern in flight_patterns_list:
                        flight_patterns_mask = flight_patterns_masks[flight_pattern.id]
                        deleted_flight_patterns_count += 1
                        if flight_patterns_mask.is_empty():
                            continue

                        flight_bases_to_keep.add(flight_pattern.flight_base_id)
                        for mask in dmm.generate_masks(flight_patterns_mask):
                            fp = ApmFlightPattern()
                            fp.id = self._min_flight_pattern_id
                            self._min_flight_pattern_id += 1
                            fp.flight_base_id = flight_pattern.flight_base_id
                            fp.marketing_carrier = flight_pattern.marketing_carrier
                            fp.marketing_carrier_iata = flight_pattern.marketing_carrier_iata
                            fp.marketing_flight_number = flight_pattern.marketing_flight_number
                            fp.bucket_key = flight_pattern.bucket_key
                            fp.flight_leg_key = flight_pattern.flight_leg_key
                            fp.operating_from = mask[0].replace('.', '-')
                            fp.operating_until = mask[1].replace('.', '-')
                            fp.operating_on_days = mask[2]
                            fp.leg_seq_number = flight_pattern.leg_seq_number
                            fp.is_administrative = flight_pattern.is_administrative
                            fp.is_codeshare = flight_pattern.is_codeshare
                            fp.performance = flight_pattern.performance
                            fp.arrival_day_shift = flight_pattern.arrival_day_shift
                            fp.operating_flight_pattern_id = flight_pattern.operating_flight_pattern_id
                            updated_flight_patterns.append(fp)

                flight_bases_to_delete = set(
                    _id for _id in existing_flight_base_ids if _id not in flight_bases_to_keep
                )

                unload_date = datetime.utcnow()
                response_data['upload_date'] = unload_date
                response_data['apm_filename'] = apm_filename
                response_data['memo'] = memo

                if not new_flight_bases or not new_flight_patterns:
                    raise Exception('No new flights are parsed from the specified file')

                response_data['deleted_old_segments'] = deleted_flight_patterns_count
                self.delete_old_flight_patterns(cursor)

                if len(flight_bases_to_delete) > 0:
                    response_data['deleted_old_flight_bases'] = len(flight_bases_to_delete)
                    self.delete_old_flight_bases(cursor, flight_bases_to_delete)

                upload_record = ApmImportedFile()
                upload_record.imported_date = unload_date
                upload_record.file_name = apm_filename
                # TODO(u-jeen): implement when authorisation is added
                upload_record.user = 'unknown'
                upload_record.memo = memo
                unique_flights = list(set(flights))
                unique_flights.sort()
                upload_record.description = '; '.join(unique_flights)

                response_data['new_flight_bases'] = len(new_flight_bases)
                self.save_new_flight_bases(cursor, new_flight_bases)
                response_data['new_segments'] = len(new_flight_patterns)
                response_data['updated_segments'] = len(updated_flight_patterns)
                self.save_new_flight_patterns(cursor, new_flight_patterns)
                self.save_new_flight_patterns(cursor, updated_flight_patterns)

                self.save_upload_record(cursor, upload_record)
                db_conn.commit()
            except Exception as e:
                tb = traceback.format_exc()
                error = '{} {}'.format(e, tb)
                logger.error('Error while uploading APM file: %s %s', e, tb)
                db_conn.rollback()

        response_data['file_size'] = file_size
        response_data['error'] = error
        response_data['messages'] = messages
        return response_data

    @staticmethod
    def verify_flight_base(fb, flight_number, segment_number):
        # type: (ApmFlightBase, str, int) -> List(str)
        errors = []
        if not ApmParser.valid_time(fb.scheduled_departure_time):
            errors.append(
                'Error: Invalid or missing departure time for flight {} segment {}'.format(
                    flight_number,
                    segment_number,
                )
            )
        if not ApmParser.valid_time(fb.scheduled_arrival_time):
            errors.append(
                'Error: Invalid or missing arrival time for flight {} segment {}'.format(
                    flight_number,
                    segment_number,
                )
            )
        return errors

    @staticmethod
    def valid_time(scheduled_time):
        return scheduled_time >= 0 and scheduled_time < 2400 and scheduled_time % 100 < 60

    def get_flight_key(self, carrier, flight_number):
        return '{}/{}'.format(carrier, flight_number)

    # date_mask - the dates when the flight (the one being parsed) is operating
    # date_mask_to_remove - all dates covered by the period_start / period_end boundaries if the flight being parsed
    def process_schedule_elem(self, schedule_el, dmm, date_mask, date_mask_to_remove, flight_number, flight_title):
        days = schedule_el.get('days', '').strip()
        period_start_date = schedule_el.get('period_start_date', '').strip()
        period_end_date = schedule_el.get('period_end_date', '').strip()
        if period_start_date or period_end_date:
            if not period_start_date:
                raise Exception(
                    'Warning: flight {} {} has period with no start date'.format(
                        flight_number,
                        flight_title,
                    )
                )
            if not period_end_date:
                raise Exception(
                    'Warning: flight {} {} has period with no end date'.format(
                        flight_number,
                        flight_title,
                    )
                )
            try:
                datetime.strptime(period_start_date, YYYYMMDD)
                datetime.strptime(period_end_date, YYYYMMDD)
            except ValueError:
                raise Exception(
                    'Warning: flight {} {} has period with invalid start/end days: {} - {}'.format(
                        flight_number,
                        flight_title,
                        period_start_date,
                        period_end_date,
                    )
                )
            if not days:
                raise Exception(
                    'Warning: flight {} {} has period {} - {} with invalid days: {}'.format(
                        flight_number,
                        flight_title,
                        period_start_date,
                        period_end_date,
                        days,
                    )
                )
            for day in days.split('|'):
                if '-' in day:
                    try:
                        datetime.strptime(day, YYYYMMDD)
                        dmm.add_date_str(day, date_mask)
                        dmm.add_date_str(day, date_mask_to_remove)
                    except ValueError:
                        raise Exception(
                            'Warning: flight {} {} has period with invalid days list: {}'.format(
                                flight_number,
                                flight_title,
                                days,
                            )
                        )
                else:
                    if not re.match('[0-7]+', day) or not int(day) or int(day) < 0:
                        raise Exception(
                            'Warning: flight {} {} has period with invalid days of week: {}'.format(
                                flight_number,
                                flight_title,
                                days,
                            )
                        )
                    dmm.add_range(period_start_date, period_end_date, int(day), date_mask)
                    dmm.add_range(period_start_date, period_end_date, 1234567, date_mask_to_remove)
        else:
            if not days or '-' not in days:
                raise Exception(
                    'Warning: flight {} {} has period with empty or invalid days: {}'.format(
                        flight_number,
                        flight_title,
                        days,
                    )
                )
            days_list = days.split('|')
            for day in days_list:
                try:
                    datetime.strptime(day, YYYYMMDD)
                    dmm.add_date_str(day, date_mask)
                    dmm.add_date_str(day, date_mask_to_remove)
                except ValueError:
                    raise Exception(
                        'Warning: flight {} {} has period with invalid days: {}'.format(
                            flight_number,
                            flight_title,
                            days,
                        )
                    )
        exclude_days = schedule_el.get('exclude_days', '').strip()
        if exclude_days:
            if '-' not in exclude_days:
                raise Exception(
                    'Warning: flight {} {} has period with invalid exclude days: {}'.format(
                        flight_number,
                        flight_title,
                        exclude_days,
                    )
                )
            exclude_days_list = exclude_days.split('|')
            for exclude_day in exclude_days_list:
                try:
                    datetime.strptime(exclude_day, YYYYMMDD)
                    dmm.remove_date_str(exclude_day, date_mask)
                    dmm.add_date_str(exclude_day, date_mask_to_remove)
                except ValueError:
                    raise Exception(
                        'Warning: flight {} {} has period with invalid exclude days: {}'.format(
                            flight_number,
                            flight_title,
                            exclude_days,
                        )
                    )

    def create_apm_flight_base(self, carrier, flight_number, segment_number, stoppoint_from, stoppoint_to):
        fb = ApmFlightBase()
        fb.id = self._min_flight_base_id
        self._min_flight_base_id += 1
        fb.operating_carrier = carrier.Id
        fb.operating_carrier_iata = carrier.Iata
        fb.operating_flight_number = flight_number[2:].strip()
        fb.itinerary_variation = fb.id
        fb.leg_seq_number = segment_number
        fb.bucket_key = '{}.{}.{}'.format(fb.operating_carrier, fb.operating_flight_number, fb.leg_seq_number)
        fb.departure_station = stoppoint_from.station_code
        fb.departure_station_iata = stoppoint_from.station_iata
        fb.scheduled_departure_time = stoppoint_from.departure_time_int
        fb.arrival_station = stoppoint_to.station_code
        fb.arrival_station_iata = stoppoint_to.station_iata
        fb.scheduled_arrival_time = stoppoint_to.arrival_time_int
        fb.local_dep_utc_time_var = 100  # unused, but null is not allowed, and 0 is valid value
        fb.local_arr_utc_time_var = 100  # unused, but null is not allowed, and 0 is valid value
        fb.departure_terminal = ''
        fb.arrival_terminal = ''
        fb.aircraft_model = ''
        fb.flying_carrier_iata = ''
        fb.intl_dom_status = ''
        fb.traffic_restriction_code = ''
        fb.designated_carrier = 0
        return fb

    def get_carrier(self, thread_el, carriers, messages, flight_number):
        carrier_code = thread_el.get('carrier_code', '').strip()
        error = False
        if carrier_code:
            carrier = carriers.get(carrier_code)
            if not carrier:
                messages.append(
                    'Warning: flight {} has a carrier with unknown carrier code {}'.format(
                        flight_number,
                        carrier_code,
                    )
                )
        if not carrier:
            carrier_title = thread_el.get('carrier_title', '').strip()
            if not carrier_title:
                error = True
                messages.append('Error: Flight with no carrier title: {}'.format(flight_number))
            carrier = carriers.get(carrier_title)
        if not carrier:
            error = True
            messages.append('Unknown carrier title: {} in flight {}'.format(carrier_title, flight_number))
        return carrier, error

    def get_stoppoints(self, stoppoints_el, flight_number, flight_title, stations, messages):
        stoppoints = []
        error = False
        for stoppoint_el in stoppoints_el.xpath('.//stoppoint'):
            try:
                stoppoint = self.get_stop_point(stoppoint_el, flight_number, flight_title, stations)
                if len(stoppoints) > 0 and stoppoint.arrival_time_int < 0 and stoppoint.departure_time_int >= 0:
                    stoppoint.arrival_time_int = subtract_five_mins(stoppoint.departure_time_int)
                stoppoints.append(stoppoint)
            except Exception as e:
                error = True
                messages.append('Error: {}'.format(e))

        if len(stoppoints) < 2:
            error = True
            messages.append(
                'Error: Flight {} {} has only {} known stations in its route'.format(
                    flight_number,
                    flight_title,
                    len(stoppoints),
                )
            )

        return stoppoints, error

    def get_stop_point(self, stoppoint_el, flight_number, flight_title, stations):
        station_code = stoppoint_el.get('station_code', '').strip()
        if station_code not in stations:
            raise Exception(
                'Unknown station code {} in flight {} {}'.format(
                    station_code,
                    flight_number,
                    flight_title,
                )
            )
        station_iata = stations[station_code].IataCode
        departure_time = stoppoint_el.get('departure_time', '').strip()
        arrival_time = stoppoint_el.get('arrival_time', '').strip()
        if not departure_time and not arrival_time:
            raise Exception(
                'No departure and no arrival time in flight {} {}'.format(
                    flight_number,
                    flight_title,
                )
            )
        departure_time_int = -1
        arrival_time_int = -1
        if departure_time:
            try:
                text_without_zeroes = departure_time.replace(':', '').lstrip('0')
                departure_time_int = int(text_without_zeroes) if text_without_zeroes else 0
            except Exception:
                raise Exception(
                    'Invalid departure time {} in flight {} {}'.format(
                        departure_time,
                        flight_number,
                        flight_title,
                    )
                )
        if arrival_time:
            try:
                text_without_zeroes = arrival_time.replace(':', '').lstrip('0')
                arrival_time_int = int(text_without_zeroes) if text_without_zeroes else 0
            except Exception:
                raise Exception(
                    'Invalid arrival time {} in flight {} {}'.format(
                        arrival_time,
                        flight_number,
                        flight_title,
                    )
                )
        return Stoppoint(station_code, station_iata, departure_time_int, arrival_time_int)

    def fetch_stations_to_dict(self, cursor):
        stations = {}
        cursor.execute(
            '''
            select
                id,
                settlement_id,
                country_id,
                time_zone_id,
                iata,
                icao,
                sirena,
                title_default
            from
                station_with_codes
            order by
                id;
            ''')
        for row in cursor:
            station = TStationWithCodes()
            station.Station.Id = row[0]
            station.Station.SettlementId = row[1]
            station.Station.CountryId = row[2]
            station.Station.TimeZoneId = row[3]
            station.IataCode = row[4]
            station.IcaoCode = row[5]
            station.SirenaCode = row[6]
            station.Station.TitleDefault = row[7]

            stations[str(station.Station.Id)] = station

        return stations

    def fetch_carriers_to_dict(self, cursor):
        carriers = {}
        cursor.execute(
            '''
            select
                id,
                iata,
                icao,
                sirena_id,
                title
            from
                carrier
            order by
                id
            ''')
        for row in cursor:
            carrier = TCarrier()
            carrier.Id = row[0]
            carrier.Iata = row[1] if row[1] else ''
            carrier.Icao = row[2] if row[2] else ''
            carrier.SirenaId = row[3] if row[3] else ''
            carrier.Title = row[4]
            carriers[str(carrier.Title)] = carrier
            carriers[str(carrier.Id)] = carrier

        return carriers

    def fetch_flight_patterns(self, cursor, dmm):
        flight_patterns_map = defaultdict(list)
        flight_patterns_masks = {}
        cursor.execute(
            '''
            select
                id,
                bucket_key,
                flight_base_id,
                flight_leg_key,
                operating_from,
                operating_until,
                operating_on_days,
                marketing_carrier,
                marketing_carrier_iata,
                marketing_flight_number,
                is_administrative,
                is_codeshare,
                performance,
                arrival_day_shift,
                operating_flight_pattern_id,
                leg_seq_number,
                designated_carrier
            from
                apm_flight_pattern
            ''')

        for row in cursor:
            fp = ApmFlightPattern()
            fp.id = row[0]
            fp.bucket_key = row[1] if row[1] else ''
            fp.flight_base_id = row[2] if row[2] else 0
            fp.flight_leg_key = row[3] if row[3] else 0
            fp.operating_from = row[4] if row[4] else ''
            fp.operating_until = row[5] if row[5] else ''
            fp.operating_on_days = row[6] if row[6] else 0
            fp.marketing_carrier = row[7] if row[7] else 0
            fp.marketing_carrier_iata = row[8] if row[8] else ''
            fp.marketing_flight_number = row[9] if row[9] else ''
            fp.is_administrative = bool(row[10]) if row[10] else False
            fp.is_codeshare = bool(row[11]) if row[11] else False
            fp.performance = row[12] if row[12] else 0
            fp.arrival_day_shift = row[13] if row[13] else 0
            fp.operating_flight_pattern_id = row[14] if row[14] else 0
            fp.leg_seq_number = row[15] if row[15] else 0
            fp.designated_carrier = row[16] if row[16] else 0

            if self._min_flight_pattern_id <= fp.id:
                self._min_flight_pattern_id = fp.id + 1

            key = self.get_flight_key(fp.marketing_carrier, fp.marketing_flight_number)
            flight_patterns_map[key].append(fp)
            fp_mask = dmm.new_date_mask()
            period_start_date = fp.operating_from.strftime(YYYYMMDD)
            period_end_date = fp.operating_until.strftime(YYYYMMDD)
            dmm.add_range(period_start_date, period_end_date, fp.operating_on_days, fp_mask)
            flight_patterns_masks[fp.id] = fp_mask

        return flight_patterns_map, flight_patterns_masks

    def fetch_flight_base_ids_to_list(self, cursor):
        flight_base_ids = []
        cursor.execute(
            '''
            select
                id
            from
                apm_flight_base
            ''')
        for row in cursor:
            fb_id = row[0]
            flight_base_ids.append(row[0])
            if self._min_flight_base_id <= fb_id:
                self._min_flight_base_id = fb_id + 1

        return flight_base_ids

    def delete_old_flight_patterns(self, cursor):
        cursor.execute('delete from apm_flight_pattern where true')

    def delete_old_flight_bases(self, cursor, flight_bases_to_delete):
        sql = 'delete from apm_flight_base where id in ({})'.format(','.join([str(x) for x in flight_bases_to_delete]))
        cursor.execute(sql)

    def save_new_flight_bases(self, cursor, new_flight_bases):
        data = []
        for fb in new_flight_bases:
            data.append((
                fb.id,
                fb.bucket_key,
                fb.operating_carrier,
                fb.operating_carrier_iata,
                fb.operating_flight_number,
                fb.itinerary_variation,
                fb.leg_seq_number,
                fb.departure_station,
                fb.departure_station_iata,
                fb.scheduled_departure_time,
                fb.local_dep_utc_time_var,
                fb.departure_terminal,
                fb.arrival_station,
                fb.arrival_station_iata,
                fb.scheduled_arrival_time,
                fb.local_arr_utc_time_var,
                fb.arrival_terminal,
                fb.aircraft_model,
                fb.flying_carrier_iata,
                fb.intl_dom_status,
                fb.traffic_restriction_code,
                fb.designated_carrier,
            ))
        columns = [
            'id',
            'bucket_key',
            'operating_carrier',
            'operating_carrier_iata',
            'operating_flight_number',
            'itinerary_variation',
            'leg_seq_number',
            'departure_station',
            'departure_station_iata',
            'scheduled_departure_time',
            'local_dep_utc_time_var',
            'departure_terminal',
            'arrival_station',
            'arrival_station_iata',
            'scheduled_arrival_time',
            'local_arr_utc_time_var',
            'arrival_terminal',
            'aircraft_model',
            'flying_carrier_iata',
            'intl_dom_status',
            'traffic_restriction_code',
            'designated_carrier',
        ]
        insert_query = 'insert into apm_flight_base ({}) values %s'.format(', '.join(columns))
        psycopg2.extras.execute_values(
            cursor, insert_query, data, template=None, page_size=100
        )

    def save_new_flight_patterns(self, cursor, new_flight_patterns):
        data = []
        for fp in new_flight_patterns:
            data.append((
                fp.id,
                fp.bucket_key,
                fp.flight_base_id,
                fp.flight_leg_key,
                fp.operating_from,
                fp.operating_until,
                fp.operating_on_days,
                fp.marketing_carrier,
                fp.marketing_carrier_iata,
                fp.marketing_flight_number,
                fp.is_administrative,
                fp.is_codeshare,
                fp.performance,
                fp.arrival_day_shift,
                fp.operating_flight_pattern_id,
                fp.leg_seq_number,
                fp.designated_carrier,
            ))
        columns = [
            'id',
            'bucket_key',
            'flight_base_id',
            'flight_leg_key',
            'operating_from',
            'operating_until',
            'operating_on_days',
            'marketing_carrier',
            'marketing_carrier_iata',
            'marketing_flight_number',
            'is_administrative',
            'is_codeshare',
            'performance',
            'arrival_day_shift',
            'operating_flight_pattern_id',
            'leg_seq_number',
            'designated_carrier',
        ]
        insert_query = 'insert into apm_flight_pattern ({}) values %s'.format(', '.join(columns))
        psycopg2.extras.execute_values(
            cursor, insert_query, data, template=None, page_size=100
        )

    def save_upload_record(self, cursor, upload_record):
        columns = [
            'imported_date',
            'file_name',
            '\"user\"',
            'memo',
            'description',
        ]
        cursor.execute(
            psycopg2.sql.SQL(
                'insert into apm_imported_file ({}) values (%s, %s, %s, %s, %s)'.format(
                    ', '.join(columns)
                )
            ),
            [
                upload_record.imported_date,
                upload_record.file_name,
                upload_record.user,
                upload_record.memo,
                upload_record.description,
            ],
        )

    def get_psycopg2_conn_string(self, config):
        conn_string = ' '.join([
            'dbname={}'.format(config.PGAAS_DATABASE_NAME),
            'user={}'.format(config.PGAAS_USER),
            'host={}'.format('c-{cluster_id}.rw.db.yandex.net'.format(cluster_id=config.PGAAS_CLUSTER_ID)),
            'port={}'.format(config.PGAAS_PORT),
            'sslmode=require',
            'password={}'.format(config.PGAAS_PASSWORD),
        ])
        return conn_string


def subtract_five_mins(int_time):
    # For 00:00 - 00:05 return midnight, don't bother changing the date
    if int_time <= 5:
        return 0
    if int_time % 100 >= 5:
        return int_time - 5
    return int_time - 45
