import os
import sqlalchemy.orm as sorm

from flask import Flask
from flask_admin import Admin
from flask_sqlalchemy import SQLAlchemy

import travel.avia.shared_flights.admin.app.config as config
from sqlalchemy_continuum import version_class
from travel.avia.shared_flights.admin.app.ping import ping_handler
from travel.avia.shared_flights.admin.app.view.apm_upload import ApmUploadView, apm_upload_handler
from travel.avia.shared_flights.admin.app.view.blacklist import BlacklistHistoryView, BlacklistView
from travel.avia.shared_flights.admin.app.view.carrier import CarrierView
from travel.avia.shared_flights.admin.app.view.flight_merge_rule import FlightMergeRuleView
from travel.avia.shared_flights.admin.app.view.flight_status_source import FlightStatusSourceView
from travel.avia.shared_flights.admin.app.view.iata_correction import IataCorrectionHistoryView, IataCorrectionView
from travel.avia.shared_flights.admin.app.view.station_status_source import StationStatusSourceView, \
    StationStatusSourceHistoryView
from travel.avia.shared_flights.admin.app.view.stop_point import StopPointView
from travel.avia.shared_flights.admin.app.view.station import StationView
from travel.avia.shared_flights.lib.python.db_models.carrier import Carrier
from travel.avia.shared_flights.lib.python.db_models.station import Station
from travel.avia.shared_flights.lib.python.db.engine import conn_string
from travel.avia.shared_flights.lib.python.db_models.apm_imported_file import ApmImportedFile
from travel.avia.shared_flights.lib.python.db_models.base import Base
from travel.avia.shared_flights.lib.python.db_models.blacklist import Blacklist
from travel.avia.shared_flights.lib.python.db_models.flight_merge_rule import FlightMergeRule
from travel.avia.shared_flights.lib.python.db_models.flight_status_source import FlightStatusSource
from travel.avia.shared_flights.lib.python.db_models.iata_correction_rule import IataCorrectionRule
from travel.avia.shared_flights.lib.python.db_models.station_status_source import StationStatusSource
from travel.avia.shared_flights.lib.python.db_models.stop_point import StopPoint


def wsgi():
    os.environ['SERVER_NAME'] = '*'
    flask_app = Flask(__name__, subdomain_matching=True)
    app_dir = os.getenv('AVIA_ADMIN_APP_DIR')
    if app_dir:
        flask_app.template_folder = os.path.join(app_dir, 'templates')

    flask_app.secret_key = r';x,/VyydAru!va4C}.W:5S*UynesB,su!2=W#Nrz0kC\\dbS+8tTrCF_%%7-d{5mk'
    flask_app.config['SQLALCHEMY_DATABASE_URI'] = conn_string(
        user=config.PGAAS_USER,
        password=config.PGAAS_PASSWORD,
        database=config.PGAAS_DATABASE_NAME,
        cluster_id=config.PGAAS_CLUSTER_ID,
        port=config.PGAAS_PORT,
        hostname=config.PGAAS_HOST_NAME,
    )
    flask_app.config['EXPLAIN_TEMPLATE_LOADING'] = True

    sorm.configure_mappers()  # required for table versioning
    db_engine = SQLAlchemy(flask_app, metadata=Base.metadata)

    if config.AVIA_SHARED_FLIGHTS_CREATE_SCHEMA:
        db_engine.create_all()

    admin = Admin(flask_app, name='shared-flights admin', template_mode='bootstrap3')
    # Add administrative views here

    admin.add_view(BlacklistView(Blacklist, db_engine.session, name='Banned'))
    admin.add_view(BlacklistHistoryView(version_class(Blacklist), db_engine.session, name='Ban(hist)'))

    admin.add_view(IataCorrectionView(IataCorrectionRule, db_engine.session, name='IATA correction'))
    admin.add_view(
        IataCorrectionHistoryView(version_class(IataCorrectionRule), db_engine.session, name='IATA corr(hist)'))

    admin.add_view(StationView(Station, db_engine.session, name='Stations'))
    admin.add_view(CarrierView(Carrier, db_engine.session, name='Carriers'))
    admin.add_view(StopPointView(StopPoint, db_engine.session, name='Stop points'))

    admin.add_view(ApmUploadView(ApmImportedFile, db_engine.session, name='APM uploads'))
    apm_upload_handler(flask_app, config)

    admin.add_view(FlightStatusSourceView(FlightStatusSource, db_engine.session, name='Status sources'))
    admin.add_view(StationStatusSourceView(StationStatusSource, db_engine.session, name='Station status source'))
    admin.add_view(StationStatusSourceHistoryView(version_class(StationStatusSource),
                                                  db_engine.session,
                                                  name='Station status source (hist)',
                                                  ))

    admin.add_view(FlightMergeRuleView(FlightMergeRule, db_engine.session, name='Merge'))

    ping_handler(flask_app, db_engine, db_engine.session, db_engine.session())

    return flask_app
