package main

import (
	"fmt"
	"net/http"

	"github.com/go-chi/chi/v5"
	"github.com/go-chi/chi/v5/middleware"

	"a.yandex-team.ru/library/go/core/log"
	arczap "a.yandex-team.ru/library/go/core/log/zap"
	aviaLogging "a.yandex-team.ru/travel/avia/library/go/logging"
	"a.yandex-team.ru/travel/avia/shared_flights/api/internal/config"
	"a.yandex-team.ru/travel/avia/shared_flights/api/internal/server"
	loadSnapshotServices "a.yandex-team.ru/travel/avia/shared_flights/api/internal/services/loadsnapshot"
	storageServices "a.yandex-team.ru/travel/avia/shared_flights/api/internal/services/storage"
	"a.yandex-team.ru/travel/avia/shared_flights/api/internal/storage"
	"a.yandex-team.ru/travel/avia/shared_flights/lib/go/logger"
)

func main() {
	configuration := config.NewConfig()
	appLog, err := configureLogging(&configuration.LoggingConfig)
	if err == nil {
		defer appLog.L.Sync()
	} else {
		logger.Logger().Warnf("logger creation failed: %s", err.Error())
	}
	configureMetrics()
	go runProfiler(ProfilerPort)

	s := server.NewServer()
	newStorage := storage.NewStorage()
	storageService := storageServices.NewStorageService(newStorage)

	logger.Logger().Debug("Loading configuration")
	newStorage.FlightStorage().SetFlightCacheConfig(configuration.FlightCacheConfig)
	logger.Logger().Debug("Loaded configuration", log.Reflect("config", &configuration))
	loadSnapshotService := loadSnapshotServices.NewLoadSnapshotService(storageService, configuration)
	s.RunServer(storageService, loadSnapshotService, configuration, APIPort)
}

func runProfiler(port int) {
	router := chi.NewRouter()
	router.Mount("/debug", middleware.Profiler())
	logger.Logger().Fatal("Server error",
		log.Error(
			http.ListenAndServe(
				fmt.Sprintf(":%d", port),
				router,
			),
		),
	)
}

func configureLogging(cfg *config.Logging) (*arczap.Logger, error) {
	converted, err := cfg.ToLoggerConfig()
	if err != nil {
		return nil, err
	}
	newLogger, err := aviaLogging.NewLogger(converted)
	if err != nil {
		return nil, err
	}
	logger.SetLogger(newLogger)
	return newLogger, nil
}
