package main

import (
	"bufio"
	"context"
	"encoding/json"
	"fmt"
	"os"
	"strings"

	"github.com/golang/protobuf/proto"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/travel/avia/shared_flights/lib/go/logger"
	"a.yandex-team.ru/travel/library/go/configuration"
	"a.yandex-team.ru/travel/library/go/dicts/base"
	"a.yandex-team.ru/travel/proto/dicts/avia"
)

type FlightRepository struct {
	Flights []*avia.TFlight
}

func NewFlightRepository() *FlightRepository {
	return &FlightRepository{}
}

func (cr *FlightRepository) Write(b []byte) (int, error) {
	carrier := &avia.TFlight{}
	if err := proto.Unmarshal(b, carrier); err != nil {
		return 0, fmt.Errorf("CarrierRepository:Write: %w", err)
	}
	cr.Flights = append(cr.Flights, carrier)
	return len(b), nil
}

type readConfig struct {
	Filename string `config:"filename,required"`
}

func main() {
	var err error

	loader := configuration.NewDefaultConfitaLoader()
	var config readConfig
	if err = loader.Load(context.Background(), &config); err != nil {
		logger.Logger().Error("Cannot load config", log.Error(err))
		return
	}
	var bytesIterator *base.BytesIterator
	if bytesIterator, err = base.BuildIteratorFromFile(config.Filename); err != nil {
		logger.Logger().Error("Cannot initialize bytes iterator", log.Error(err))
		return
	}
	flightRepo := NewFlightRepository()
	if err = bytesIterator.Populate(flightRepo); err != nil {
		logger.Logger().Error("Cannot populate repository")
		return
	}

	logger.Logger().Info(
		"Read all the flights into repository",
		log.Int("count", len(flightRepo.Flights)),
	)

	for {
		fmt.Println("Ask me for a flight")
		var flightNumber string
		scanner := bufio.NewScanner(os.Stdin)
		scanner.Scan()
		flightNumber = scanner.Text()
		fmt.Printf("You've entered a flight %s\n", flightNumber)
		for _, flight := range flightRepo.Flights {
			if strings.Contains(flight.Title, flightNumber) {
				marshalled, _ := json.MarshalIndent(flight, " ", " ")
				fmt.Println(string(marshalled))
			}
		}

	}
}
