package main

import (
	"context"
	"strings"
	"time"

	"github.com/heetch/confita"
	"github.com/heetch/confita/backend/env"
	"github.com/heetch/confita/backend/flags"

	"a.yandex-team.ru/library/go/core/log"
	storageServices "a.yandex-team.ru/travel/avia/shared_flights/api/internal/services/storage"
	"a.yandex-team.ru/travel/avia/shared_flights/api/internal/services/storage/flight"
	"a.yandex-team.ru/travel/avia/shared_flights/api/pkg/structs"
	"a.yandex-team.ru/travel/avia/shared_flights/lib/go/dtutil"
	"a.yandex-team.ru/travel/avia/shared_flights/lib/go/logger"
	"a.yandex-team.ru/yt/go/ypath"
	"a.yandex-team.ru/yt/go/yt"
	"a.yandex-team.ru/yt/go/yt/ythttp"
)

type tFlightNumber struct {
	carrier int32
	number  string
}

type EnvConfig struct {
	YtProxy   string `config:"AVIA_YT_PROXY"`
	YtToken   string `config:"AVIA_OAUTH_TOKEN"`
	StartDate string `config:"AVIA_DUMP_START_DATE"`
	EndDate   string `config:"AVIA_DUMP_END_DATE"`
	YtPath    string `config:"AVIA_DUMP_YT_PATH"`
}

const YtProxy = "hahn.yt.yandex.net"

type Record struct {
	MarketingCarrierID     int32  `yson:"marketing_carrier_id"`
	MarketingCarrierCode   string `yson:"marketing_carrier_code"`
	MarketingFlightNumber  string `yson:"marketing_flight_number"`
	SegmentNumber          int8   `yson:"segment_number"`
	SegmentCount           int8   `yson:"segment_count"`
	AirportFromID          int32  `yson:"airport_from_id"`
	AirportToID            int32  `yson:"airport_to_id"`
	FlightDate             string `yson:"flight_date"`
	SegmentDepartureDate   string `yson:"segment_departure_date"`
	SegmentArrivalDate     string `yson:"segment_arrival_date"`
	DepartureTime          string `yson:"departure_time"`
	DepartureTimeUTC       string `yson:"departure_time_utc"`
	ArrivalTime            string `yson:"arrival_time"`
	ArrivalTimeUTC         string `yson:"arrival_time_utc"`
	DepartureTerminal      string `yson:"departure_terminal"`
	ArrivalTerminal        string `yson:"arrival_terminal"`
	TransportModel         string `yson:"transport_model"`
	TransportModelID       int32  `yson:"transport_model_id"`
	FilingCarrierID        int32  `yson:"filing_carrier_id"`
	IsCodeshare            bool   `yson:"is_codeshare"`
	Domestic               string `yson:"domestic"`
	OperatingCarrierID     int32  `yson:"operating_carrier_id"`
	OperatingCarrierCode   string `yson:"operating_carrier_code"`
	OperatingFlightNumber  string `yson:"operating_flight_number"`
	OperatingSegmentNumber int8   `yson:"operating_segment_number"`
	DataSource             string `yson:"data_source"`
	RecordSource           string `yson:"record_source"`
}

func DumpYt(storage *storageServices.ServiceInstance) {
	type countIsCodeshare struct{}
	type countIsDop struct{}
	type countIsOperating struct{}
	type countFlightsCount struct{}
	type countSegmentsCount struct{}
	type countFlightsOnDateCount struct{}
	type countFlightDataErrors struct{}
	type countFlightDataErrors404 struct{}
	type countNoFlight struct{}
	type countGoodFlight struct{}

	sourceToString := make(map[structs.FlightBaseSource]string)
	sourceToString[structs.UnknownSource] = "unknown"
	sourceToString[structs.InnovataSource] = "innovata"
	sourceToString[structs.SirenaSource] = "sirena"
	sourceToString[structs.ApmSource] = "arm"
	sourceToString[structs.DopSource] = "dop"
	sourceToString[structs.AmadeusSource] = "amadeus"

	var counters = make(map[interface{}]int)
	flightNumbers := make(map[tFlightNumber]bool)
	for _, v := range storage.Storage().FlightStorage().GetFlightPatterns() {
		if v.IsCodeshare {
			counters[countIsCodeshare{}] += 1
		} else {
			counters[countIsOperating{}] += 1
		}

		flightNumbers[tFlightNumber{
			v.MarketingCarrier,
			v.MarketingFlightNumber,
		}] = true
	}

	for _, v := range storage.Storage().FlightStorage().GetDopFlightPatterns() {
		counters[countIsDop{}] += 1

		flightNumbers[tFlightNumber{
			v.MarketingCarrier,
			v.MarketingFlightNumber,
		}] = true
	}

	ctx := context.Background()
	loader := confita.NewLoader(
		env.NewBackend(),
		flags.NewBackend(),
	)
	yesterday := string(dtutil.FormatDateIso(time.Now().Add(-24 * time.Hour)))
	envConfig := EnvConfig{
		YtProxy:   YtProxy,
		StartDate: yesterday,
		EndDate:   yesterday,
	}
	err := loader.Load(ctx, &envConfig)
	if err != nil {
		logger.Logger().Fatal("Cannot load configuration", log.Error(err))
	}
	if len(envConfig.YtPath) == 0 || !strings.HasSuffix(envConfig.YtPath, "/") {
		logger.Logger().Fatal("Please specify AVIA_DUMP_YT_PATH that ends with slash")
	}
	ytConfig := yt.Config{
		Proxy: envConfig.YtProxy,
		Token: envConfig.YtToken,
	}

	startDateIndex := dtutil.DateCache.IndexOfStringDateP(dtutil.StringDate(envConfig.StartDate))
	endDateIndex := dtutil.DateCache.IndexOfStringDateP(dtutil.StringDate(envConfig.EndDate))

	for dateIndex := startDateIndex; dateIndex <= endDateIndex; dateIndex++ {
		currentDate := string(dtutil.DateCache.Date(dateIndex).StringDateDashed())
		flightCountsPerSource := make(map[string]int)
		logger.Logger().Info("Start processing date", log.String("current-date", currentDate))

		wctx, cancel := context.WithCancel(ctx)
		defer cancel()
		ytClient, _ := ythttp.NewClient(&ytConfig)
		path := ypath.Path(envConfig.YtPath + currentDate)
		tx, err := ytClient.BeginTx(wctx, nil)
		if err != nil {
			logger.Logger().Fatal("Unable to access yt", log.Error(err))
		}
		tableExists, err := tx.NodeExists(wctx, path, nil)
		if err != nil {
			logger.Logger().Fatal("Unable to test if yt table exists", log.Error(err))
		}
		if tableExists {
			err := tx.RemoveNode(wctx, path, nil)
			if err != nil {
				logger.Logger().Fatal("Unable to remove existing yt table", log.Error(err))
			}
		}
		_, err = tx.CreateNode(wctx, path, yt.NodeTable, &yt.CreateNodeOptions{
			Recursive: true,
			Force:     true,
		})
		if err != nil {
			logger.Logger().Fatal("Unable to create yt node", log.Error(err))
		}
		tw, err := tx.WriteTable(wctx, path, nil)
		if err != nil {
			logger.Logger().Fatal("Unable to create yt table", log.Error(err))
		}

		errors := []error{}
		for k := range flightNumbers {
			counters[countFlightsCount{}] += 1
			flightData, err := storage.GetFlightData(
				flight.NewCarrierParamByID(k.carrier),
				k.number,
				currentDate,
				nil,
				"",
				false,
				false,
				false,
			)
			if err != nil {
				counters[countFlightDataErrors{}] += 1
				if strings.Contains(err.Error(), "error 404") {
					counters[countFlightDataErrors404{}] += 1
				}
				if len(errors) < 10 && !strings.Contains(err.Error(), "error 404") {
					errors = append(errors, err)
				}
				continue
			}
			if len(flightData) == 0 {
				counters[countNoFlight{}] += 1
				continue
			}
			counters[countGoodFlight{}] += 1
			segmentCount := len(flightData)
			counters[countSegmentsCount{}] += segmentCount
			for _, flightSegment := range flightData {
				dataSource, ok := sourceToString[flightSegment.FlightBase.Source]
				if !ok {
					dataSource = "unknown"
				}
				departure := flightSegment.ScheduledDeparture()
				var departureDay, departureTime, departureTimeUTC string
				if !departure.IsZero() {
					departureDay = departure.Format(dtutil.IsoDate)
					departureTime = departure.Format(dtutil.IsoTime)
					departureTimeUTC = dtutil.FormatDateTimeISO(departure.In(time.UTC))
				}

				arrival := flightSegment.ScheduledArrival()
				var arrivalDay, arrivalTime, arrivalTimeUTC string
				if !arrival.IsZero() {
					arrivalDay = arrival.Format(dtutil.IsoDate)
					arrivalTime = arrival.Format(dtutil.IsoTime)
					arrivalTimeUTC = dtutil.FormatDateTimeISO(arrival.In(time.UTC))
				}

				transportModelID := int32(flightSegment.FlightBase.AircraftTypeID)

				record := Record{
					MarketingCarrierID:     flightSegment.FlightPattern.MarketingCarrier,
					MarketingCarrierCode:   flightSegment.FlightPattern.MarketingCarrierCode,
					MarketingFlightNumber:  flightSegment.FlightPattern.MarketingFlightNumber,
					SegmentNumber:          int8(flightSegment.FlightPattern.LegNumber),
					SegmentCount:           int8(segmentCount),
					AirportFromID:          int32(flightSegment.FlightBase.DepartureStation),
					AirportToID:            int32(flightSegment.FlightBase.ArrivalStation),
					FlightDate:             currentDate,
					SegmentDepartureDate:   departureDay,
					SegmentArrivalDate:     arrivalDay,
					DepartureTime:          departureTime,
					DepartureTimeUTC:       departureTimeUTC,
					ArrivalTime:            arrivalTime,
					ArrivalTimeUTC:         arrivalTimeUTC,
					DepartureTerminal:      flightSegment.FlightBase.DepartureTerminal,
					ArrivalTerminal:        flightSegment.FlightBase.ArrivalTerminal,
					TransportModel:         storage.Storage().TransportModels().GetCode(transportModelID),
					TransportModelID:       transportModelID,
					FilingCarrierID:        flightSegment.FlightPattern.FilingCarrier,
					IsCodeshare:            flightSegment.FlightPattern.IsCodeshare,
					Domestic:               flightSegment.FlightBase.IntlDomesticStatus,
					OperatingCarrierID:     flightSegment.FlightBase.OperatingCarrier,
					OperatingCarrierCode:   flightSegment.FlightBase.OperatingCarrierCode,
					OperatingFlightNumber:  flightSegment.FlightBase.OperatingFlightNumber,
					OperatingSegmentNumber: int8(flightSegment.FlightBase.LegNumber),
					DataSource:             dataSource,
					RecordSource:           "shared-flights",
				}
				err = tw.Write(record)
				if err != nil {
					logger.Logger().Fatal("Unable to write data to yt table", log.Error(err))
				}
				flightCountsPerSource[dataSource] += 1
			}
		}

		err = tw.Commit()
		if err != nil {
			logger.Logger().Fatal("Unable to flush data to yt table", log.Error(err))
		}
		err = tx.Commit()
		if err != nil {
			logger.Logger().Fatal("Unable to commit yt transaction", log.Error(err))
		}

		for k, v := range counters {
			logger.Logger().Infof("%T: %v", k, v)
		}

		for k, v := range flightCountsPerSource {
			logger.Logger().Infof("Flights count: %v: %v", k, v)
		}

		for _, err := range errors {
			logger.Logger().Info("Flight error: ", log.Error(err))
		}
	}
	logger.Logger().Info("Done iterating over flight patterns")
}
