package config

import (
	"context"
	"time"

	"github.com/heetch/confita"
	"github.com/heetch/confita/backend/env"
	"github.com/heetch/confita/backend/flags"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/travel/avia/shared_flights/lib/go/logger"
)

type APIConfig struct {
	AllowSlowQueries bool `config:"AVIA_ALLOW_SLOW_QUERIES"`
	P2P              P2PConfig
}

type P2PConfig struct {
	MaxP2PStationsInRequest int `config:"AVIA_MAX_P2P_STATIONS_IN_REQUEST"`
}

// Config for AWS/MDS/S3 services
type AwsConfig struct {
	Endpoint                 string
	Region                   string
	MaxRetries               int
	Bucket                   string `config:"AVIA_MDS_BUCKET"`      // Bucket to write data
	BucketFolder             string `config:"AVIA_MDS_BUCKET_PATH"` // Folder in bucket where all data for api is written
	LatestFileName           string // This file contains the latest markers for the base and delta snapshots
	BaseSnapshotFileName     string // Base snapshot file name
	DeltaSnapshotFilePattern string // Delta snapshot file name pattern
}

type SnapshotWorkerConfig struct {
	SleepBetweenReloads time.Duration `config:"AVIA_LOAD_SNAPSHOT_WORKER_SLEEP_BETWEEN_RELOADS"`
	MaxSuccessiveErrors int           `config:"AVIA_LOAD_SNAPSHOT_WORKER_MAX_SUCCESSIVE_ERRORS"`
}

type FlightCacheConfig struct {
	// Only load these carriers, ignore the rest _must_ be used together with max-flights
	OnlyCarriers []string `config:"only-carriers"`
	// Limit the number of flights; only-carriers flag has a higher priority than this one
	MaxFlights int32 `config:"max-flights"`
	// OnlyCarriers converted to map
	OnlyCarriersMap map[string]bool `config:"-"`
}

// Config for the server
type Config struct {
	APIConfig            APIConfig
	AwsConfig            AwsConfig
	SnapshotWorkerConfig SnapshotWorkerConfig
	FlightCacheConfig    FlightCacheConfig
	LoggingConfig        Logging
}

func NewConfig() *Config {
	loader := confita.NewLoader(
		env.NewBackend(),
		flags.NewBackend(),
	)
	config := &Config{
		APIConfig: APIConfig{
			AllowSlowQueries: false,
			P2P: P2PConfig{
				MaxP2PStationsInRequest: 100,
			},
		},
		AwsConfig: AwsConfig{
			Endpoint:                 "https://s3.mds.yandex.net",
			Region:                   "yandex",
			MaxRetries:               2,
			Bucket:                   "avia-shared-flights-dev",
			BucketFolder:             "snapshots/dev",
			LatestFileName:           "latest.txt",
			BaseSnapshotFileName:     "snapshot.zip",
			DeltaSnapshotFilePattern: "delta-%v.zip", // "%v" is substituted with the current delta marker
		},
		SnapshotWorkerConfig: SnapshotWorkerConfig{
			SleepBetweenReloads: time.Second * 5,
			MaxSuccessiveErrors: 10,
		},
		FlightCacheConfig: FlightCacheConfig{
			OnlyCarriers:    []string{},
			OnlyCarriersMap: map[string]bool{},
			MaxFlights:      0,
		},
		LoggingConfig: Logging{
			LogLevel:       log.DebugString,
			AppLogFileName: "api.log",
		},
	}

	err := loader.Load(context.Background(), config)
	if err != nil {
		logger.Logger().Fatal("Cannot load configuration", log.Error(err))
	}

	if len(config.FlightCacheConfig.OnlyCarriers) > 0 {
		carriersMap := map[string]bool{}
		for _, carrier := range config.FlightCacheConfig.OnlyCarriers {
			carriersMap[carrier] = true
		}
		config.FlightCacheConfig.OnlyCarriersMap = carriersMap
	}

	return config
}
