package handlers

import (
	"fmt"
	"net/http"
	"strconv"
	"strings"

	"github.com/labstack/echo/v4"
	"go.uber.org/zap"

	"a.yandex-team.ru/library/go/core/xerrors"
	"a.yandex-team.ru/travel/avia/shared_flights/api/internal/handlers/lib"
	"a.yandex-team.ru/travel/avia/shared_flights/api/internal/services/storage"
	"a.yandex-team.ru/travel/avia/shared_flights/api/internal/services/storage/accessors"
	"a.yandex-team.ru/travel/avia/shared_flights/lib/go/dtutil"
	"a.yandex-team.ru/travel/avia/shared_flights/lib/go/httputil"
	"a.yandex-team.ru/travel/avia/shared_flights/lib/go/strutil"
)

type storageAccessorHandler struct {
	service *storage.Service
}

func (h *storageAccessorHandler) GetFlightBase(c echo.Context) error {
	flightBaseID, err := strconv.ParseInt(c.Param("id"), 10, 32)

	if err != nil {
		return c.JSON(http.StatusBadRequest, err.Error())
	}

	value, storageErr := h.service.Instance().GetFlightBase(int32(flightBaseID), false)

	if storageErr != nil {
		return c.JSON(http.StatusNotFound, storageErr.Error())
	}

	return c.JSON(http.StatusOK, value)
}

func (h *storageAccessorHandler) GetFlightBaseRoute() string {
	return "/flight-base/:id/"
}

func (h *storageAccessorHandler) GetDopFlightBase(c echo.Context) error {
	flightBaseID, err := strconv.ParseInt(c.Param("id"), 10, 32)

	if err != nil {
		return c.JSON(http.StatusBadRequest, err.Error())
	}

	value, storageErr := h.service.Instance().GetFlightBase(int32(flightBaseID), true)

	if storageErr != nil {
		return c.JSON(http.StatusNotFound, storageErr.Error())
	}

	return c.JSON(http.StatusOK, value)
}

func (h *storageAccessorHandler) GetDopFlightBaseRoute() string {
	return "/dop-flight-base/:id/"
}

func (h *storageAccessorHandler) GetFlightPattern(c echo.Context) error {
	flightPatternID, err := strconv.ParseInt(c.Param("id"), 10, 32)

	if err != nil {
		return c.JSON(http.StatusBadRequest, zap.Error(err))
	}

	value, storageErr := h.service.Instance().GetFlightPattern(int32(flightPatternID), false)

	if storageErr != nil {
		return c.JSON(http.StatusNotFound, fmt.Sprintf("%v", storageErr))
	}

	return c.JSON(http.StatusOK, value)
}

func (h *storageAccessorHandler) GetFlightPatternRoute() string {
	return "/flight-pattern/:id/"
}

func (h *storageAccessorHandler) GetDopFlightPattern(c echo.Context) error {
	flightPatternID, err := strconv.ParseInt(c.Param("id"), 10, 32)

	if err != nil {
		return c.JSON(http.StatusBadRequest, zap.Error(err))
	}

	value, storageErr := h.service.Instance().GetFlightPattern(int32(flightPatternID), true)

	if storageErr != nil {
		return c.JSON(http.StatusNotFound, fmt.Sprintf("%v", storageErr))
	}

	return c.JSON(http.StatusOK, value)
}

func (h *storageAccessorHandler) GetDopFlightPatternRoute() string {
	return "/dop-flight-pattern/:id/"
}

func (h *storageAccessorHandler) GetFlightStatus(c echo.Context) error {
	bucketKey := httputil.Unescape(c.Param("bucket_key"))
	date := dtutil.StringDate(c.Param("date")).ToIntDate()

	if date <= 0 {
		return c.JSON(
			http.StatusBadRequest,
			fmt.Sprintf("Invalid date: %v, expected format: YYYY-MM-DD", c.Param("date")))
	}

	value, storageErr := h.service.Instance().GetFlightStatus(bucketKey, date)

	if storageErr != nil {
		return c.JSON(http.StatusBadRequest, fmt.Sprintf("%v", storageErr))
	}

	return c.JSON(http.StatusOK, value)
}

func (h *storageAccessorHandler) GetFlightStatusRoute() string {
	return "/flight-status/:bucket_key/:date/"
}

func (h *storageAccessorHandler) GetRawFlight(c echo.Context) error {
	service := h.service.Instance()
	carrierText := httputil.Unescape(c.Param("carrier"))
	marketingCarrier, err := strconv.ParseInt(carrierText, 10, 32)
	flightNumber := httputil.Unescape(strutil.RemoveLeadingZeroes(c.Param("flight")))
	isDop := true
	if c.QueryParam("show_dop") != "" {
		isDop = strings.EqualFold(c.QueryParam("show_dop"), "true")
	}
	isText := strings.EqualFold(c.QueryParam("text"), "true")

	var marketingCarriers []int32
	if err != nil {
		marketingCarriers = service.GetCarriersByCode(carrierText)
	} else {
		marketingCarriers = []int32{int32(marketingCarrier)}
	}

	var result []accessors.FlightPatternAndBase
	for _, carrier := range marketingCarriers {
		value, storageErr := service.GetRawFlight(int32(carrier), flightNumber, isDop)

		if storageErr == nil {
			result = append(result, value...)
		}
	}

	if len(result) == 0 {
		return c.JSON(http.StatusNotFound, xerrors.Errorf("unknown flight %v/%v", carrierText, flightNumber))
	}

	if isText {
		return c.String(http.StatusOK, ConvertToText(result))
	}

	return c.JSON(http.StatusOK, result)
}

func ConvertToText(flights []accessors.FlightPatternAndBase) string {
	lines := make([]string, 0, len(flights))
	currentLegNumber := int32(1)
	for _, flight := range flights {
		line := strings.Join(
			[]string{
				flight.FlightPattern.MarketingCarrierCode,
				flight.FlightPattern.MarketingFlightNumber,
				fmt.Sprintf("%v", flight.FlightPattern.LegNumber),
				flight.FlightBase.DepartureStationCode,
				dtutil.IntTime(flight.FlightBase.DepartureTimeScheduled).String()[:5],
				overnightToString(flight.FlightPattern.DepartureDayShift),
				"-",
				flight.FlightBase.ArrivalStationCode,
				dtutil.IntTime(flight.FlightBase.ArrivalTimeScheduled).String()[:5],
				overnightToString(flight.FlightPattern.ArrivalDayShift),
				flight.FlightPattern.OperatingFromDate,
				"-",
				flight.FlightPattern.OperatingUntilDate,
				dtutil.OperatingDays(flight.FlightPattern.OperatingOnDays).ReadableString(),
				fmt.Sprintf("%v", flight.FlightPattern.IsCodeshare),
				fmt.Sprintf("%v", flight.FlightBase.Source),
			},
			" ",
		)
		if currentLegNumber != flight.FlightPattern.LegNumber {
			currentLegNumber = flight.FlightPattern.LegNumber
			lines = append(lines, " ")
		}
		lines = append(lines, line)
	}
	return strings.Join(lines, "\n")
}

func overnightToString(value int32) string {
	if value == 0 {
		return "  "
	}
	return fmt.Sprintf("+%v", value)
}

func (h *storageAccessorHandler) GetRawFlightRoute() string {
	return "/flight-raw/:carrier/:flight/"
}

func (h *storageAccessorHandler) GetRawCodeshareFlights(c echo.Context) error {
	service := h.service.Instance()
	carrierText := httputil.Unescape(c.Param("carrier"))
	marketingCarrier, err := strconv.ParseInt(carrierText, 10, 32)
	flightNumber := httputil.Unescape(strutil.RemoveLeadingZeroes(c.Param("flight")))

	var marketingCarriers []int32
	if err != nil {
		marketingCarriers = service.GetCarriersByCode(carrierText)
	} else {
		marketingCarriers = []int32{int32(marketingCarrier)}
	}

	result := []string{}
	for _, carrier := range marketingCarriers {
		flights, ok := service.Storage().FlightStorage().GetCodeshareFlightKeys(carrier, flightNumber)
		if !ok {
			continue
		}
		for _, flight := range flights {
			result = append(result, convertCarrier(flight, service))
		}
	}

	return c.JSON(http.StatusOK, result)
}

func convertCarrier(flight string, service *storage.ServiceInstance) string {
	parts := strings.Split(flight, "/")
	if len(parts) < 2 {
		return flight
	}
	carrierID, err := strconv.ParseInt(parts[0], 10, 32)
	if err != nil {
		return flight
	}
	return fmt.Sprintf("%v(%v)/%v", service.GetCarrierCodeByID(int32(carrierID)), parts[0], strings.Join(parts[1:], "/"))
}

func (h *storageAccessorHandler) GetRawCodeshareFlightsRoute() string {
	return "/flight-codeshares/:carrier/:flight/"
}

func (h *storageAccessorHandler) GetRawP2PFlights(c echo.Context) error {
	service := h.service.Instance()
	fromParam := httputil.Unescape(c.Param("from"))
	if len(fromParam) == 0 {
		return c.JSON(http.StatusBadRequest, "Missing \"from\" station")
	}
	stationFrom, err := lib.ParseStation(fromParam, service.Stations())
	if err != nil {
		return c.JSON(http.StatusNotFound, xerrors.Errorf("cannot get \"from\" station: %w", err).Error())
	}
	tz := service.GetTimeZoneByStationID(int64(stationFrom.Station.Id))
	if tz == nil {
		return c.JSON(
			http.StatusInternalServerError,
			xerrors.Errorf("cannot get station (%v) timezone", stationFrom.Station.Id).Error(),
		)
	}

	toParam := httputil.Unescape(c.Param("to"))
	if len(toParam) == 0 {
		return c.JSON(http.StatusBadRequest, "Missing \"to\" station")
	}
	stationTo, err := lib.ParseStation(toParam, service.Stations())
	if err != nil {
		return c.JSON(http.StatusNotFound, xerrors.Errorf("cannot get \"to\" station: %w", err).Error())
	}
	tz = service.GetTimeZoneByStationID(int64(stationTo.Station.Id))
	if tz == nil {
		return c.JSON(
			http.StatusInternalServerError,
			xerrors.Errorf("cannot get station (%v) timezone", stationTo.Station.Id).Error(),
		)
	}

	result, err := service.GetRawP2PFlights(int64(stationFrom.Station.Id), int64(stationTo.Station.Id))
	if err != nil {
		return c.JSON(
			http.StatusNotFound,
			fmt.Sprintf("%+v", xerrors.Errorf("no flights found from %+v to %+v; error: %w", stationFrom, stationTo, err)),
		)
	}

	return c.JSON(http.StatusOK, result)
}

func (h *storageAccessorHandler) GetRawP2PFlightsRoute() string {
	return "/flight-p2p-raw/:from/:to/"
}

func (h *storageAccessorHandler) GetIataCorrectionRules(c echo.Context) error {
	result, err := h.service.Instance().GetIataCorrectionRules()
	if err != nil {
		return c.JSON(http.StatusNotFound, xerrors.Errorf("no IATA correction rules found"))
	}

	return c.JSON(http.StatusOK, result)
}

func (h *storageAccessorHandler) GetIataCorrectionRulesRoute() string {
	return "/iata-correction-rules/"
}

func (h *storageAccessorHandler) GetCarrier(c echo.Context) error {
	service := h.service.Instance()

	carrierText := httputil.Unescape(c.Param("carrier"))
	carrierID, err := strconv.ParseInt(carrierText, 10, 64)

	if err != nil {
		return c.JSON(http.StatusOK, service.GetCarriersByCode(carrierText))
	}
	return c.JSON(http.StatusOK, service.GetCarrierCodeByID(int32(carrierID)))
}

func (h *storageAccessorHandler) GetCarrierRoute() string {
	return "/carrier/:carrier/"
}

func (h *storageAccessorHandler) GetFlightStatusSources(c echo.Context) error {
	result := h.service.Instance().GetFlightStatusSources()
	return c.JSON(http.StatusOK, result)
}

func (h *storageAccessorHandler) GetFlightStatusSourcesRoute() string {
	return "/flight-status-sources/"
}

func (h *storageAccessorHandler) GetFlightMergeRules(c echo.Context) error {
	result := h.service.Instance().GetFlightMergeRules()
	return c.JSON(http.StatusOK, result)
}

func (h *storageAccessorHandler) GetFlightMergeRulesRoute() string {
	return "/flight-merge-rules/"
}

func (h *storageAccessorHandler) GetPopularityScores(c echo.Context) error {
	service := h.service.Instance()

	return c.JSON(http.StatusOK, service.GetCarriersPopularityScores(c.QueryParam("national_version")))
}

func (h *storageAccessorHandler) GetPopularityScoresRoute() string {
	return "/carriers-popularity-scores/"
}

func (h *storageAccessorHandler) GetPopularCarriersForFlight(c echo.Context) error {
	service := h.service.Instance()

	return c.JSON(http.StatusOK, service.GetPopularCarriersForFlight(c.Param("flight-number"), c.QueryParam("national_version")))
}

func (h *storageAccessorHandler) GetPopularCarriersForFlightRoute() string {
	return "/carriers-popular/:flight-number/"
}

func NewStorageAccessorHandler(service *storage.Service) *storageAccessorHandler {
	return &storageAccessorHandler{
		service: service,
	}
}
