package handlers

import (
	"fmt"
	"net/http"
	"strings"
	"time"

	"github.com/labstack/echo/v4"

	"a.yandex-team.ru/library/go/core/xerrors"
	"a.yandex-team.ru/travel/avia/shared_flights/api/internal/handlers/lib"
	"a.yandex-team.ru/travel/avia/shared_flights/api/internal/services/storage"
	"a.yandex-team.ru/travel/avia/shared_flights/api/internal/services/storage/aeroflot_variants/format"
	"a.yandex-team.ru/travel/avia/shared_flights/api/internal/utils"
	"a.yandex-team.ru/travel/avia/shared_flights/lib/go/dtutil"
	"a.yandex-team.ru/travel/proto/shared_flights/snapshots"
)

type aeroflotConnectingVariantsHandler struct {
	service *storage.Service
}

const (
	MaxFromStations = 5
	MaxToStations   = 5
	MaxFromToPairs  = 10
	MaxPeriodInDays = 31.
)

func (h *aeroflotConnectingVariantsHandler) Handle(c echo.Context) error {
	service := h.service.Instance()

	form := AeroflotConnectingVariantsForm{}
	if err := form.Fill(c, service); err.Error != nil {
		return c.JSON(err.HTTPCode, err.Error.Error())
	}

	value, err := service.GetAeroflotConnectingVariants(
		form.departFrom,
		form.arriveTo,
		form.after,
		form.before,
		form.isOneWay,
		form.nationalVersion,
		form.showBanned,
	)

	if err != nil {
		if httpErr, ok := err.(*utils.ErrorWithHTTPCode); ok {
			return c.JSON(httpErr.HTTPCode, fmt.Sprintf("%v", httpErr))
		}
		return c.JSON(http.StatusInternalServerError, fmt.Sprintf("%v", err))
	}

	if form.isText {
		return c.String(http.StatusOK, h.convertToText(value))
	}

	return c.JSON(http.StatusOK, value)
}

func (h *aeroflotConnectingVariantsHandler) GetRoute() string {
	return "/aeroflot-variants/"
}

func NewAeroflotConnectingVariantsHandler(service *storage.Service) *aeroflotConnectingVariantsHandler {
	return &aeroflotConnectingVariantsHandler{
		service: service,
	}
}

func (h *aeroflotConnectingVariantsHandler) convertToText(response format.Response) string {
	lines := make([]string, 0, len(response.Variants))
	for _, variant := range response.Variants {
		line := strings.Join(
			[]string{
				h.getSliceText(variant.Forward),
				" </> ",
				h.getSliceText(variant.Backward),
				getBannedText(variant.Banned),
			},
			" ",
		)
		lines = append(lines, line)
	}

	result := make([]string, 0, len(lines))
	lastLine := ""
	for lineNumber, line := range lines {
		lastLineParts := strings.Split(lastLine, "/")
		lineParts := strings.Split(line, "/")
		var curLine string
		if len(lineParts) != len(lastLineParts) {
			curLine = line
		} else {
			curLineParts := []string{}
			for partNum := 0; partNum < len(lastLineParts); partNum++ {
				if lineParts[partNum] != lastLineParts[partNum] {
					curLineParts = append(curLineParts, lineParts[partNum])
				} else {
					curLineParts = append(curLineParts, strings.Repeat(" ", len(lineParts[partNum])))
				}
			}
			curLine = strings.Join(curLineParts, "/")
		}
		result = append(
			result,
			strings.Join(
				[]string{
					fmt.Sprintf("%5d", lineNumber+1),
					curLine,
				},
				" ",
			),
		)
		lastLine = line
	}

	return strings.Join(result, "\n")
}

func (h *aeroflotConnectingVariantsHandler) getSliceText(slice format.Slice) string {
	if len(slice.Flights) == 0 {
		return ""
	}
	secondFlightText := []string{strings.Repeat(" ", 30)}
	flightsSeparator := " "
	if len(slice.Flights) > 1 {
		secondFlightText = h.getFlightText(slice.Flights[1])
		flightsSeparator = "/"
	}
	var result []string
	result = append(result, h.getFlightText(slice.Flights[0])...)
	result = append(result, flightsSeparator)
	result = append(result, secondFlightText...)
	result = append(result, fmt.Sprintf("%4d", slice.Duration))
	return strings.Join(result, " ")
}

func (h *aeroflotConnectingVariantsHandler) getFlightText(flight format.Flight) []string {
	return []string{
		fmt.Sprintf("%7s", flight.Title),
		fmt.Sprintf("%3s", h.service.Instance().Stations().GetCode(int64(flight.DepartureStation))),
		strings.ReplaceAll(flight.DepartureDatetime[5:16], "T", " "),
		"-",
		strings.ReplaceAll(flight.ArrivalDatetime[5:16], "T", " "),
		fmt.Sprintf("%3s", h.service.Instance().Stations().GetCode(int64(flight.ArrivalStation))),
	}
}

func getBannedText(isBanned string) string {
	if isBanned == "true" {
		return "X"
	}
	return " "
}

type AeroflotConnectingVariantsForm struct {
	departFrom      []*snapshots.TStationWithCodes
	arriveTo        []*snapshots.TStationWithCodes
	after           string
	before          string
	nationalVersion string
	isOneWay        bool
	showBanned      bool
	isText          bool
}

func (f *AeroflotConnectingVariantsForm) Fill(
	c echo.Context,
	service *storage.ServiceInstance,
) lib.ErrorWithCode {
	params := c.QueryParams()
	fromParam := params["from"]
	if len(fromParam) == 0 {
		return lib.ErrorWithCode{
			HTTPCode: http.StatusBadRequest,
			Error:    xerrors.New("Missing \"from\" station"),
		}
	}
	if len(fromParam) > MaxFromStations {
		return lib.ErrorWithCode{
			HTTPCode: http.StatusBadRequest,
			Error:    xerrors.Errorf("too many \"from\" stations, max %d is allowed", MaxFromStations),
		}
	}
	toParam := params["to"]
	if len(toParam) == 0 {
		return lib.ErrorWithCode{
			HTTPCode: http.StatusBadRequest,
			Error:    xerrors.New("Missing \"to\" station"),
		}
	}
	if len(toParam) > MaxToStations {
		return lib.ErrorWithCode{
			HTTPCode: http.StatusBadRequest,
			Error:    xerrors.Errorf("too many \"to\" stations, max %d is allowed", MaxToStations),
		}
	}
	if len(fromParam)*len(toParam) > MaxFromToPairs {
		return lib.ErrorWithCode{
			HTTPCode: http.StatusBadRequest,
			Error:    xerrors.Errorf("too many station pairs, max %d is allowed", MaxFromToPairs),
		}
	}

	from := []*snapshots.TStationWithCodes{}
	for _, station := range fromParam {
		if len(station) == 0 {
			return lib.ErrorWithCode{
				HTTPCode: http.StatusBadRequest,
				Error:    xerrors.New("Empty \"from\" station"),
			}
		}
		stationFrom, err := lib.ParseStation(station, service.StationProvider())
		if err != nil {
			return lib.ErrorWithCode{
				HTTPCode: http.StatusNotFound,
				Error:    xerrors.Errorf("cannot get \"to\" station: %w", err),
			}
		}
		tz := service.GetTimeZoneByStationID(int64(stationFrom.Station.Id))
		if tz == nil {
			return lib.ErrorWithCode{
				HTTPCode: http.StatusInternalServerError,
				Error:    xerrors.Errorf("cannot get station (%v) timezone", stationFrom.Station.Id),
			}
		}
		from = append(from, stationFrom)
	}

	to := []*snapshots.TStationWithCodes{}
	for _, station := range toParam {
		if len(station) == 0 {
			return lib.ErrorWithCode{
				HTTPCode: http.StatusBadRequest,
				Error:    xerrors.New("Empty \"to\" station"),
			}
		}
		stationTo, err := lib.ParseStation(station, service.StationProvider())
		if err != nil {
			return lib.ErrorWithCode{
				HTTPCode: http.StatusNotFound,
				Error:    xerrors.Errorf("cannot get \"to\" station: %w", err),
			}
		}
		tz := service.GetTimeZoneByStationID(int64(stationTo.Station.Id))
		if tz == nil {
			return lib.ErrorWithCode{
				HTTPCode: http.StatusInternalServerError,
				Error:    xerrors.Errorf("cannot get station (%v) timezone", stationTo.Station.Id),
			}
		}
		to = append(to, stationTo)
	}

	afterDateParam := c.QueryParam("after")
	var afterDate time.Time
	var err error
	if len(afterDateParam) > 0 {
		afterDate, err = time.Parse(dtutil.IsoDate, afterDateParam)
		if err != nil {
			return lib.ErrorWithCode{
				HTTPCode: http.StatusBadRequest,
				Error:    xerrors.Errorf("Invalid \"after\" date %+v, error: %v", afterDate, err),
			}
		}
	} else {
		afterDate = time.Now().AddDate(0, 0, 1)
	}
	// handles the case of dates like 2020-06-31 which is interpeted as 2020-07-01
	f.after = string(dtutil.FormatDateIso(afterDate))

	beforeDateParam := c.QueryParam("before")
	var beforeDate time.Time
	if len(beforeDateParam) > 0 {
		beforeDate, err = time.Parse(dtutil.IsoDate, beforeDateParam)
		if err != nil {
			return lib.ErrorWithCode{
				HTTPCode: http.StatusBadRequest,
				Error:    xerrors.Errorf("Invalid \"before\" date %+v, error: %v", beforeDate, err),
			}
		}
	} else {
		beforeDate = afterDate.AddDate(0, 0, 7)
	}
	// handles the case of dates like 2020-06-31 which is interpeted as 2020-07-01
	f.before = string(dtutil.FormatDateIso(beforeDate))

	rangeDuration := beforeDate.Sub(afterDate).Hours()
	if rangeDuration > MaxPeriodInDays*24. {
		return lib.ErrorWithCode{
			HTTPCode: http.StatusBadRequest,
			Error:    xerrors.Errorf("Range %v - %v is too wide, max %v days is allowed", f.after, f.before, MaxPeriodInDays),
		}
	}

	f.departFrom = from
	f.arriveTo = to
	f.nationalVersion = c.QueryParam("national_version")
	f.showBanned = strings.EqualFold(c.QueryParam("show_banned"), "true")
	f.isOneWay = strings.EqualFold(c.QueryParam("one_way"), "true")
	f.isText = strings.EqualFold(c.QueryParam("text"), "true")

	return lib.ErrorWithCode{
		HTTPCode: http.StatusOK,
		Error:    nil,
	}
}
