package handlers

import (
	"net/http"

	"github.com/labstack/echo/v4"

	"a.yandex-team.ru/library/go/core/xerrors"
	"a.yandex-team.ru/travel/avia/shared_flights/api/internal/handlers/lib"
	"a.yandex-team.ru/travel/avia/shared_flights/api/internal/services/storage"
	"a.yandex-team.ru/travel/avia/shared_flights/lib/go/httputil"
	"a.yandex-team.ru/travel/proto/shared_flights/snapshots"
)

type aeroflotConnectionsHandler struct {
	service *storage.Service
}

func (h *aeroflotConnectionsHandler) Handle(c echo.Context) error {
	service := h.service.Instance()

	form := AeroflotConnectionsForm{}
	if err := form.Fill(c, service); err.Error != nil {
		return c.JSON(err.HTTPCode, err.Error.Error())
	}

	value := service.Storage().FlightStorage().AeroflotCache().GetConnectionsMap(
		int64(form.departFrom.Station.Id),
		int64(form.arriveTo.Station.Id),
	)
	return c.JSON(http.StatusOK, value)
}

func (h *aeroflotConnectionsHandler) GetRoute() string {
	return "/aeroflot-connections/:from/:to/"
}

func NewAeroflotConnectionsHandler(service *storage.Service) *aeroflotConnectionsHandler {
	return &aeroflotConnectionsHandler{
		service: service,
	}
}

type AeroflotConnectionsForm struct {
	departFrom *snapshots.TStationWithCodes
	arriveTo   *snapshots.TStationWithCodes
}

func (f *AeroflotConnectionsForm) Fill(
	c echo.Context,
	service *storage.ServiceInstance,
) lib.ErrorWithCode {
	textFrom := httputil.Unescape(c.Param("from"))
	if len(textFrom) == 0 {
		return lib.ErrorWithCode{
			HTTPCode: http.StatusBadRequest,
			Error:    xerrors.New("Missing \"from\" station"),
		}
	}
	textTo := httputil.Unescape(c.Param("to"))
	if len(textTo) == 0 {
		return lib.ErrorWithCode{
			HTTPCode: http.StatusBadRequest,
			Error:    xerrors.New("Missing \"to\" station"),
		}
	}

	stationFrom, err := lib.ParseStation(textFrom, service.StationProvider())
	if err != nil {
		return lib.ErrorWithCode{
			HTTPCode: http.StatusNotFound,
			Error:    xerrors.Errorf("cannot get \"from\" station: %w", err),
		}
	}

	stationTo, err := lib.ParseStation(textTo, service.StationProvider())
	if err != nil {
		return lib.ErrorWithCode{
			HTTPCode: http.StatusNotFound,
			Error:    xerrors.Errorf("cannot get \"to\" station: %w", err),
		}
	}

	f.departFrom = stationFrom
	f.arriveTo = stationTo

	return lib.ErrorWithCode{
		HTTPCode: http.StatusOK,
		Error:    nil,
	}
}
