package handlers

import (
	"fmt"
	"net/http"
	"strings"
	"time"

	"github.com/labstack/echo/v4"

	"a.yandex-team.ru/travel/avia/shared_flights/api/internal/handlers/lib"
	"a.yandex-team.ru/travel/avia/shared_flights/api/internal/services/storage"
	"a.yandex-team.ru/travel/avia/shared_flights/api/internal/services/storage/flight"
	"a.yandex-team.ru/travel/avia/shared_flights/api/internal/utils"
	"a.yandex-team.ru/travel/avia/shared_flights/lib/go/dtutil"
	"a.yandex-team.ru/travel/avia/shared_flights/lib/go/httputil"
	"a.yandex-team.ru/travel/avia/shared_flights/lib/go/strutil"
	"a.yandex-team.ru/travel/proto/shared_flights/snapshots"
)

type FlightHandler struct {
	service *storage.Service
}

func (h *FlightHandler) Handle(c echo.Context) error {
	var err error
	service := h.service.Instance()

	carrierText := httputil.Unescape(c.Param("carrier"))
	flightNumber := httputil.Unescape(strutil.RemoveLeadingZeroes(c.Param("flight")))
	departureDate := c.Param("departureDate")

	var fromStation *snapshots.TStationWithCodes = nil
	if len(c.QueryParam("from")) > 0 {
		fromStation, err = lib.ParseStation(c.QueryParam("from"), service.StationProvider())
		if err != nil {
			return c.JSON(http.StatusNotFound, fmt.Sprintf("%v", err))
		}
	}

	depDateValue, departureDateErr := time.Parse(dtutil.IsoDate, departureDate)
	if departureDateErr != nil {
		return c.JSON(http.StatusBadRequest, fmt.Sprintf("Invalid date %+v, error: %v", departureDate, departureDateErr))
	}
	// handles the case of dates like 2020-06-31 which is interpeted as 2020-07-01
	departureDate = string(dtutil.FormatDateIso(depDateValue))

	nowDate, err := dtutil.DateTimeStringToTimeUTC(c.QueryParam("now"))
	if err != nil {
		return c.JSON(http.StatusBadRequest, fmt.Sprintf("%v", err))
	}
	if nowDate.IsZero() {
		nowDate = time.Now()
	}
	showBanned := strings.EqualFold(c.QueryParam("show_banned"), "true")
	nationalVersion := c.QueryParam("national_version")
	textFormat := strings.EqualFold(c.QueryParam("text"), "true")
	includeCodeshares := strings.EqualFold(c.QueryParam("codeshares"), "true")

	value, storageErr := service.GetFlight(
		carrierText, flightNumber, departureDate, fromStation, nationalVersion, showBanned, nowDate, false, includeCodeshares)

	if storageErr != nil {
		if httpErr, ok := storageErr.(*utils.ErrorWithHTTPCode); ok {
			return c.JSON(httpErr.HTTPCode, fmt.Sprintf("%v", httpErr))
		}
		return c.JSON(http.StatusInternalServerError, fmt.Sprintf("%v", storageErr))
	}

	if textFormat {
		return c.String(http.StatusOK, ConvertFlightToText(value))
	}

	if includeCodeshares {
		return c.JSON(
			http.StatusOK,
			flight.FlightSegmentWithCodeshares{
				FlightSegment: *value,
				Codeshares:    value.FlightCodeshares,
			},
		)
	}
	return c.JSON(http.StatusOK, value)
}

func (h *FlightHandler) GetRoute() string {
	return "/flight/:carrier/:flight/:departureDate/"
}

func NewFlightHandler(service *storage.Service) *FlightHandler {
	return &FlightHandler{
		service: service,
	}
}
