package handlers

import (
	"fmt"
	"net/http"
	"strconv"
	"strings"
	"time"

	"github.com/labstack/echo/v4"

	"a.yandex-team.ru/library/go/core/xerrors"
	"a.yandex-team.ru/travel/avia/shared_flights/api/internal/handlers/lib"
	"a.yandex-team.ru/travel/avia/shared_flights/api/internal/services/storage"
	"a.yandex-team.ru/travel/avia/shared_flights/api/internal/utils"
	dir "a.yandex-team.ru/travel/avia/shared_flights/lib/go/direction"
	"a.yandex-team.ru/travel/avia/shared_flights/lib/go/dtutil"
	"a.yandex-team.ru/travel/avia/shared_flights/lib/go/httputil"
	"a.yandex-team.ru/travel/avia/shared_flights/lib/go/strutil"
	"a.yandex-team.ru/travel/proto/shared_flights/snapshots"
)

type FlightBoardFormParser interface {
	FillForm(c echo.Context) (form FlightBoardForm, formErr error)
}

type flightBoardHandler struct {
	service *storage.Service
}

type FlightBoardForm struct {
	station         *snapshots.TStationWithCodes
	after           time.Time
	before          time.Time
	limit           int
	sortAscending   bool
	direction       dir.Direction
	terminal        string
	nationalVersion string
	flight          string
	formErr         error
}

func (h *flightBoardHandler) Handle(c echo.Context) error {

	form, formErr := h.FillForm(c)
	if formErr != nil {
		if httpErr, ok := formErr.(*utils.ErrorWithHTTPCode); ok {
			return c.JSON(httpErr.HTTPCode, fmt.Sprintf("%v", httpErr))
		}
		return c.JSON(http.StatusInternalServerError, fmt.Sprintf("%v", formErr))
	}

	value, storageErr := h.service.Instance().GetFlightBoard(
		form.station,
		form.after,
		form.before,
		form.limit,
		form.sortAscending,
		form.direction,
		form.terminal,
		form.nationalVersion,
		form.flight,
		time.Now(),
	)

	if storageErr != nil {
		if httpErr, ok := storageErr.(*utils.ErrorWithHTTPCode); ok {
			return c.JSON(httpErr.HTTPCode, fmt.Sprintf("%v", httpErr))
		}
		return c.JSON(http.StatusInternalServerError, fmt.Sprintf("%v", storageErr))
	}

	return c.JSON(http.StatusOK, value)
}

func (h *flightBoardHandler) FillForm(c echo.Context) (
	form FlightBoardForm,
	formErr error,
) {

	var (
		after, before, afterUtc, beforeUtc time.Time
		err                                error
	)
	service := h.service.Instance()
	stationParam := httputil.Unescape(c.Param("station"))
	if len(stationParam) == 0 {
		formErr = &utils.ErrorWithHTTPCode{
			HTTPCode:     http.StatusBadRequest,
			ErrorMessage: xerrors.Errorf("missing \"station\" parameter").Error(),
		}
		return
	}
	form.station, err = lib.ParseStation(stationParam, service.StationProvider())
	if err != nil {
		formErr = &utils.ErrorWithHTTPCode{
			HTTPCode:     http.StatusNotFound,
			ErrorMessage: xerrors.Errorf("cannot get station: %w", err).Error(),
		}
		return
	}

	stationTz := service.GetTimeZoneByStationID(int64(form.station.Station.Id))
	if stationTz == nil {
		formErr = &utils.ErrorWithHTTPCode{
			HTTPCode:     http.StatusInternalServerError,
			ErrorMessage: xerrors.Errorf("cannot get station (%v) timezone", form.station.Station.Id).Error(),
		}
		return
	}

	after, err = dtutil.StringDateTime(c.QueryParam("after")).ToTime(stationTz)
	if err != nil {
		formErr = &utils.ErrorWithHTTPCode{
			HTTPCode:     http.StatusBadRequest,
			ErrorMessage: xerrors.Errorf("invalid parameter 'after': %w", err).Error(),
		}
		return
	}
	before, err = dtutil.StringDateTime(c.QueryParam("before")).ToTime(stationTz)
	if err != nil {
		formErr = &utils.ErrorWithHTTPCode{
			HTTPCode:     http.StatusBadRequest,
			ErrorMessage: xerrors.Errorf("invalid parameter 'before': %w", err).Error(),
		}
		return
	}

	afterUtc, err = dtutil.StringDateTime(c.QueryParam("after_utc")).ToTime(time.UTC)
	if err != nil {
		formErr = &utils.ErrorWithHTTPCode{
			HTTPCode:     http.StatusBadRequest,
			ErrorMessage: xerrors.Errorf("invalid parameter 'after_utc': %w", err).Error(),
		}
		return
	}

	beforeUtc, err = dtutil.StringDateTime(c.QueryParam("before_utc")).ToTime(time.UTC)
	if err != nil {
		formErr = &utils.ErrorWithHTTPCode{
			HTTPCode:     http.StatusBadRequest,
			ErrorMessage: xerrors.Errorf("invalid parameter 'before_utc': %w", err).Error(),
		}
		return
	}

	if after.IsZero() {
		after = afterUtc
	}

	if before.IsZero() {
		before = beforeUtc
	}

	form.sortAscending = true
	if after.IsZero() && before.IsZero() {
		form.after = time.Now().Add(-time.Hour)
		form.before = time.Now().Add(24 * time.Hour)
	} else if after.IsZero() && !before.IsZero() {
		form.sortAscending = false
		form.after = before.Add(-24 * time.Hour)
		form.before = before
	} else if !after.IsZero() && before.IsZero() {
		form.after = after
		form.before = after.Add(24 * time.Hour)
	} else {
		form.after = after
		form.before = before
	}

	if form.after.After(form.before) {
		formErr = &utils.ErrorWithHTTPCode{
			HTTPCode:     http.StatusBadRequest,
			ErrorMessage: fmt.Sprintf("invalid date range: (%+v) - (%+v)", form.after, form.before),
		}
	}

	form.limit = 0
	limitStr := c.QueryParam("limit")
	if len(limitStr) > 0 {
		if form.limit, err = strconv.Atoi(limitStr); err != nil {
			formErr = &utils.ErrorWithHTTPCode{
				HTTPCode:     http.StatusBadRequest,
				ErrorMessage: xerrors.Errorf("invalid parameter 'flights_count': %w", err).Error(),
			}
			return
		}
	}

	directionString := strings.ToLower(c.QueryParam("direction"))
	form.direction = dir.DEPARTURE
	if len(directionString) > 0 {
		if form.direction, err = dir.FromString(directionString); err != nil {
			formErr = &utils.ErrorWithHTTPCode{
				HTTPCode:     http.StatusBadRequest,
				ErrorMessage: xerrors.Errorf("invalid parameter 'direction': %w", err).Error(),
			}
			return
		}
	}

	form.terminal = c.QueryParam("terminal")
	form.nationalVersion = c.QueryParam("national_version")
	form.flight = strutil.RemoveLeadingZeroes(c.QueryParam("flight"))

	return
}

func (h *flightBoardHandler) GetRoute() string {
	return "/flight-board/:station/"
}

func NewFlightBoardHandler(service *storage.Service) *flightBoardHandler {
	return &flightBoardHandler{
		service: service,
	}
}
