package handlers

import (
	"fmt"
	"net/http"
	"strings"
	"time"

	"github.com/labstack/echo/v4"

	"a.yandex-team.ru/travel/avia/shared_flights/api/internal/config"
	"a.yandex-team.ru/travel/avia/shared_flights/api/internal/services/storage"
	"a.yandex-team.ru/travel/avia/shared_flights/api/internal/services/storage/flight"
	"a.yandex-team.ru/travel/avia/shared_flights/lib/go/dtutil"
)

type FlightListHandler struct {
	service   *storage.Service
	apiConfig config.APIConfig
}

type FlightListHandlerForm struct {
	Flights string `json:"flight" form:"flight" query:"flight"`
}

func (h *FlightListHandler) Handle(c echo.Context) error {
	service := h.service.Instance()

	form := FlightListHandlerForm{}
	err := c.Bind(&form)

	if err != nil {
		return c.JSON(http.StatusBadRequest, err.Error())
	}

	if len(form.Flights) == 0 {
		return c.JSON(http.StatusBadRequest, "Missing required \"flight\" parameter")
	}

	flightTextElems := strings.Split(form.Flights, ",")
	flights := make([]flight.FlightParam, 0, len(flightTextElems))
	for _, elem := range flightTextElems {
		runes := []rune(elem)
		if len(runes) < 11 {
			return c.JSON(http.StatusBadRequest, fmt.Sprintf("Invalid flight: %v", elem))
		}
		depDateValue, departureDateErr := time.Parse(dtutil.IsoDateNoDashes, string(runes[0:8]))
		if departureDateErr != nil {
			return c.JSON(
				http.StatusBadRequest,
				fmt.Sprintf("Invalid date in flight %v, error: %+v", elem, departureDateErr),
			)
		}
		carrierCode := string(runes[8:10])
		flightNumber := strings.TrimLeft(string(runes[10:]), "0")
		if len(flightNumber) == 0 || len(flightNumber) > 5 {
			return c.JSON(http.StatusBadRequest, fmt.Sprintf("Flight has an invalid flight number: %v", elem))
		}
		// handles the case of dates like 2020-06-31 which is interpeted as 2020-07-01
		departureDate := string(dtutil.FormatDateIso(depDateValue))
		flights = append(
			flights,
			flight.FlightParam{
				CarrierCode:   carrierCode,
				FlightNumber:  flightNumber,
				DepartureDate: departureDate,
			},
		)
	}

	if !h.apiConfig.AllowSlowQueries && len(flights) > 100 {
		return c.JSON(http.StatusBadRequest, fmt.Sprintf("Too many flights in request: %v", len(flights)))
	}

	showBanned := strings.EqualFold(c.QueryParam("show_banned"), "true")
	nationalVersion := c.QueryParam("national_version")
	textFormat := strings.EqualFold(c.QueryParam("text"), "true")
	includeCodeshares := strings.EqualFold(c.QueryParam("codeshares"), "true")

	nowDate, err := dtutil.DateTimeStringToTimeUTC(c.QueryParam("now"))
	if err != nil {
		return c.JSON(http.StatusBadRequest, fmt.Sprintf("%+v", err))
	}
	if nowDate.IsZero() {
		nowDate = time.Now()
	}

	value := service.GetFlights(flights, nationalVersion, showBanned, nowDate, includeCodeshares)

	if textFormat {
		var result []string
		for _, resultElem := range value {
			result = append(result, ConvertFlightToText(resultElem))
		}
		return c.String(http.StatusOK, strings.Join(result, "\n"))
	}

	return c.JSON(http.StatusOK, value)
}

func (h *FlightListHandler) GetRoute() string {
	return "/flights/"
}

func NewFlightListHandler(service *storage.Service, apiConfig config.APIConfig) *FlightListHandler {
	return &FlightListHandler{
		service:   service,
		apiConfig: apiConfig,
	}
}
