package handlers

import (
	"net/http"

	"github.com/labstack/echo/v4"

	"a.yandex-team.ru/library/go/core/xerrors"
	"a.yandex-team.ru/travel/avia/shared_flights/api/internal/services/storage"
)

type FlightNumbersToCarriersHandler struct {
	service *storage.Service
}

type FlightNumbersToCarriersForm struct {
	FlightNumbers []string `json:"flight_numbers" form:"flight_numbers" query:"flight_numbers"`
}

func (f *FlightNumbersToCarriersForm) Validate() error {
	if len(f.FlightNumbers) == 0 {
		return xerrors.Errorf(
			"empty request. Use flight_numbers as query parameter in GET request " +
				"or {\"flight_numbers\"} json in POST request")
	}
	return nil
}

func (h *FlightNumbersToCarriersHandler) Handle(c echo.Context) error {
	var mapping = make(map[string]*int32)
	form := FlightNumbersToCarriersForm{}
	err := c.Bind(&form)

	if err != nil {
		return c.JSON(http.StatusBadRequest, err.Error())
	}

	if err = form.Validate(); err != nil {
		return c.JSON(http.StatusBadRequest, err.Error())
	}

	flightMap := h.service.Instance().GetCarriersByFlightNumbers(form.FlightNumbers)
	for k, v := range flightMap {
		if v == nil {
			mapping[k] = nil
			continue
		}
		carrierID := v.CarrierID
		mapping[k] = &carrierID

	}
	return c.JSON(200, mapping)
}

func (h *FlightNumbersToCarriersHandler) GetRoute() string {
	return "/flight-numbers-to-carriers/"
}

func (h *FlightNumbersToCarriersHandler) AmbiguityHandler(c echo.Context) error {
	return c.JSON(200, h.service.Instance().CarrierService.GetCodeAmbiguityMap())
}

func (h *FlightNumbersToCarriersHandler) GetAmbiguityRoute() string {
	return "/flight-numbers-to-carriers-ambiguity/"
}

func NewFlightNumbersToCarrierHandler(service *storage.Service) *FlightNumbersToCarriersHandler {
	return &FlightNumbersToCarriersHandler{service: service}
}
