package handlers

import (
	"net/http"
	"time"

	"github.com/labstack/echo/v4"

	"a.yandex-team.ru/library/go/core/xerrors"
	"a.yandex-team.ru/travel/avia/shared_flights/api/internal/services/storage"
	"a.yandex-team.ru/travel/avia/shared_flights/api/internal/services/storage/flight"
	"a.yandex-team.ru/travel/avia/shared_flights/lib/go/direction"
)

type FlightRangeAccessibleHandler struct {
	service *storage.Service
}

type flightRangeAccessibleForm struct {
	FlightNumbers   []string `json:"flight_numbers" form:"flight_numbers" query:"flight_numbers"`
	NationalVersion string   `json:"national_version" form:"national_version" query:"national_version"`
}

func (f *flightRangeAccessibleForm) Validate() error {
	if len(f.FlightNumbers) == 0 {
		return xerrors.Errorf(
			"empty request. Use flight_numbers as query parameter in GET request " +
				"or {\"flight_numbers\"} json in POST request")
	}
	if f.NationalVersion == "" {
		f.NationalVersion = "ru"
	}
	return nil
}

type flightRangeAccessibleResponse map[string]flightRangeAccessibleResponseItemValue

type flightRangeAccessibleResponseItemValue struct {
	Valid bool
}

func (h *FlightRangeAccessibleHandler) Handle(c echo.Context) error {
	var flightValidMap = make(flightRangeAccessibleResponse)
	form := flightRangeAccessibleForm{}
	err := c.Bind(&form)

	if err != nil {
		return c.JSON(http.StatusBadRequest, err.Error())
	}

	if err = form.Validate(); err != nil {
		return c.JSON(http.StatusBadRequest, err.Error())
	}
	service := h.service.Instance()
	flightsMap := service.GetCarriersByFlightNumbers(form.FlightNumbers)

	for k, v := range flightsMap {
		if v == nil {
			flightValidMap[k] = flightRangeAccessibleResponseItemValue{Valid: false}
			continue
		}
		segments, err := service.GetFlightRange(
			flight.NewCarrierParamByText(v.CarrierCode),
			v.FlightNumber,
			form.NationalVersion,
			false,
			time.Now(),
			1,
			1,
			direction.DEPARTURE,
			time.Now(),
		)
		if err != nil {
			flightValidMap[k] = flightRangeAccessibleResponseItemValue{Valid: false}
			continue
		}
		if len(segments) == 0 {
			flightValidMap[k] = flightRangeAccessibleResponseItemValue{Valid: false}
			continue
		}
		flightValidMap[k] = flightRangeAccessibleResponseItemValue{Valid: true}
	}
	return c.JSON(200, flightValidMap)
}

func (h *FlightRangeAccessibleHandler) GetRoute() string {
	return "/flight-range-accessible/"
}

func NewFlightRangeAccessibleHandler(service *storage.Service) *FlightRangeAccessibleHandler {
	return &FlightRangeAccessibleHandler{service: service}
}
