package handlers

import (
	"fmt"
	"net/http"
	"strconv"
	"strings"
	"time"

	"github.com/labstack/echo/v4"

	"a.yandex-team.ru/library/go/core/xerrors"
	"a.yandex-team.ru/travel/avia/shared_flights/api/internal/config"
	"a.yandex-team.ru/travel/avia/shared_flights/api/internal/handlers/lib"
	"a.yandex-team.ru/travel/avia/shared_flights/api/internal/services/storage"
	flightp2p "a.yandex-team.ru/travel/avia/shared_flights/api/internal/services/storage/flight_p2p"
	"a.yandex-team.ru/travel/avia/shared_flights/api/internal/utils"
	"a.yandex-team.ru/travel/avia/shared_flights/lib/go/dtutil"
	"a.yandex-team.ru/travel/avia/shared_flights/lib/go/strutil"
	"a.yandex-team.ru/travel/proto/shared_flights/snapshots"
)

type flightP2PHandler struct {
	service                 *storage.Service
	maxP2PStationsInRequest int
}

func (h *flightP2PHandler) Handle(c echo.Context) error {
	var err error
	service := h.service.Instance()
	params := c.QueryParams()
	fromParam, ok := params["from"]
	if !ok || len(fromParam) == 0 {
		return c.JSON(http.StatusBadRequest, "Missing \"from\" station")
	}
	toParam, ok := params["to"]
	if !ok || len(toParam) == 0 {
		return c.JSON(http.StatusBadRequest, "Missing \"to\" station")
	}

	if len(fromParam)*len(toParam) > h.maxP2PStationsInRequest {
		return c.JSON(
			http.StatusBadRequest,
			fmt.Sprintf(
				"Too many stations: %dx%d=%d max %d",
				len(fromParam),
				len(fromParam),
				len(fromParam)*len(fromParam),
				h.maxP2PStationsInRequest,
			),
		)
	}

	now := time.Now()

	from := []*snapshots.TStationWithCodes{}
	for _, station := range fromParam {
		if len(station) == 0 {
			return c.JSON(http.StatusBadRequest, "Empty \"from\" station")
		}
		stationFrom, err := lib.ParseStation(station, service.StationProvider())
		if err != nil {
			return c.JSON(http.StatusNotFound, xerrors.Errorf("cannot get \"from\" station: %w", err).Error())
		}
		tz := service.GetTimeZoneByStationID(int64(stationFrom.Station.Id))
		if tz == nil {
			return c.JSON(
				http.StatusInternalServerError,
				xerrors.Errorf("cannot get station (%v) timezone", stationFrom.Station.Id).Error(),
			)
		}
		from = append(from, stationFrom)
	}

	to := []*snapshots.TStationWithCodes{}
	for _, station := range toParam {
		if len(station) == 0 {
			return c.JSON(http.StatusBadRequest, "Empty \"to\" station")
		}
		stationTo, err := lib.ParseStation(station, service.StationProvider())
		if err != nil {
			return c.JSON(http.StatusNotFound, xerrors.Errorf("cannot get \"to\" station: %w", err).Error())
		}
		tz := service.GetTimeZoneByStationID(int64(stationTo.Station.Id))
		if tz == nil {
			return c.JSON(
				http.StatusInternalServerError,
				xerrors.Errorf("cannot get station (%v) timezone", stationTo.Station.Id).Error(),
			)
		}
		to = append(to, stationTo)
	}

	afterIsLocalTime := false
	after, err := dtutil.StringDateTime(c.QueryParam("after")).ToTime(time.UTC)
	if err != nil {
		return c.JSON(http.StatusBadRequest, xerrors.Errorf("invalid parameter 'after': %w", err).Error())
	}
	if after.IsZero() {
		afterUtc, err := dtutil.StringDateTime(c.QueryParam("after_utc")).ToTime(time.UTC)
		if err != nil {
			return c.JSON(http.StatusBadRequest, xerrors.Errorf("invalid parameter 'after_utc': %w", err).Error())
		}
		after = afterUtc
	} else {
		afterIsLocalTime = true
	}

	beforeIsLocalTime := false
	before, err := dtutil.StringDateTime(c.QueryParam("before")).ToTime(time.UTC)
	if err != nil {
		return c.JSON(http.StatusBadRequest, xerrors.Errorf("invalid parameter 'before': %w", err).Error())
	}
	if before.IsZero() {
		beforeUtc, err := dtutil.StringDateTime(c.QueryParam("before_utc")).ToTime(time.UTC)
		if err != nil {
			return c.JSON(http.StatusBadRequest, xerrors.Errorf("invalid parameter 'before_utc': %w", err).Error())
		}
		before = beforeUtc
	} else {
		beforeIsLocalTime = true
	}

	if after.IsZero() && before.IsZero() {
		after = now.Add(-time.Hour)
		before = now.Add(24 * time.Hour)
	} else if after.IsZero() && !before.IsZero() {
		after = before.Add(-24 * time.Hour)
		afterIsLocalTime = beforeIsLocalTime
	} else if !after.IsZero() && before.IsZero() {
		before = after.Add(24 * time.Hour)
		beforeIsLocalTime = afterIsLocalTime
	}

	showBanned := strings.EqualFold(c.QueryParam("show_banned"), "true")
	nationalVersion := c.QueryParam("national_version")

	if after.After(before) {
		return c.JSON(http.StatusBadRequest, fmt.Sprintf("invalid date range: (%+v) - (%+v)", after, before))
	}

	flights, ok := params["flight"]
	if !ok {
		flights = []string{}
	} else {
		for idx := range flights {
			flights[idx] = strutil.RemoveLeadingZeroes(flights[idx])
		}
	}
	flightKeys := make([]flightp2p.FlightKey, 0)
	for _, flight := range flights {
		if flight == "" {
			continue
		}
		parts := strings.Split(flight, "-")
		if len(parts) != 2 {
			return c.JSON(
				http.StatusBadRequest,
				fmt.Sprintf("\"flight\" parameter should contain exactly one dash: %v", flight),
			)
		}
		carrierText := parts[0]
		carrierID, err := strconv.ParseInt(carrierText, 10, 32)
		var carriers []int32
		if err != nil {
			carriers = service.GetCarriersByCode(carrierText)
		} else {
			carriers = []int32{int32(carrierID)}
		}
		for _, carrier := range carriers {
			flightKeys = append(
				flightKeys,
				flightp2p.FlightKey{
					MarketingCarrier: carrier,
					FlightNumber:     parts[1],
				},
			)
		}
	}

	afterParam := flightp2p.TimeParam{
		Value:   after,
		IsLocal: afterIsLocalTime,
	}
	beforeParam := flightp2p.TimeParam{
		Value:   before,
		IsLocal: beforeIsLocalTime,
	}
	value, storageErr := service.GetFlightsP2P(from, to, afterParam, beforeParam, nationalVersion, showBanned, flightKeys)

	if storageErr != nil {
		if httpErr, ok := storageErr.(*utils.ErrorWithHTTPCode); ok {
			return c.JSON(httpErr.HTTPCode, fmt.Sprintf("%v", httpErr))
		}
		return c.JSON(http.StatusInternalServerError, fmt.Sprintf("%v", storageErr))
	}

	return c.JSON(http.StatusOK, value)
}

func (h *flightP2PHandler) GetRoute() string {
	return "/flight-p2p/"
}

func NewFlightP2PHandler(service *storage.Service, config config.P2PConfig) *flightP2PHandler {
	return &flightP2PHandler{
		service:                 service,
		maxP2PStationsInRequest: config.MaxP2PStationsInRequest,
	}
}
