package handlers

import (
	"fmt"
	"net/http"
	"strconv"
	"strings"
	"time"

	"github.com/labstack/echo/v4"

	"a.yandex-team.ru/library/go/core/xerrors"
	"a.yandex-team.ru/travel/avia/shared_flights/api/internal/config"
	"a.yandex-team.ru/travel/avia/shared_flights/api/internal/handlers/lib"
	"a.yandex-team.ru/travel/avia/shared_flights/api/internal/services/storage"
	flightp2p "a.yandex-team.ru/travel/avia/shared_flights/api/internal/services/storage/flight_p2p"
	"a.yandex-team.ru/travel/avia/shared_flights/api/internal/services/storage/flight_p2p/format"
	"a.yandex-team.ru/travel/avia/shared_flights/api/internal/utils"
	"a.yandex-team.ru/travel/avia/shared_flights/lib/go/dtutil"
	"a.yandex-team.ru/travel/proto/shared_flights/snapshots"
)

type flightP2PByDateHandler struct {
	service                 *storage.Service
	maxP2PStationsInRequest int
}

func (h *flightP2PByDateHandler) Handle(c echo.Context) error {
	var err error
	service := h.service.Instance()
	params := c.QueryParams()
	fromParam := params["from"]
	if len(fromParam) == 0 {
		return c.JSON(http.StatusBadRequest, "Missing \"from\" station")
	}
	toParam := params["to"]
	if len(toParam) == 0 {
		return c.JSON(http.StatusBadRequest, "Missing \"to\" station")
	}

	if len(fromParam)*len(toParam) > h.maxP2PStationsInRequest {
		return c.JSON(
			http.StatusBadRequest,
			fmt.Sprintf(
				"Too many stations: %dx%d=%d max %d",
				len(fromParam),
				len(fromParam),
				len(fromParam)*len(fromParam),
				h.maxP2PStationsInRequest,
			),
		)
	}

	from, httpStatus, err := h.parseStationsFromParam(fromParam, "from")
	if err != nil {
		return c.JSON(httpStatus, err.Error())
	}

	to, httpStatus, err := h.parseStationsFromParam(toParam, "to")
	if err != nil {
		return c.JSON(httpStatus, err.Error())
	}

	forwardFlightDateParam := dtutil.StringDate(c.QueryParam("forward"))
	_, ok := dtutil.DateCache.IndexOfStringDate(forwardFlightDateParam)
	if !ok {
		return c.JSON(http.StatusBadRequest, xerrors.Errorf("invalid parameter 'forward': %s", forwardFlightDateParam).Error())
	}

	backwardFlightDateParam := dtutil.StringDate(c.QueryParam("backward"))
	if len(backwardFlightDateParam) > 0 {
		_, ok := dtutil.DateCache.IndexOfStringDate(backwardFlightDateParam)
		if !ok {
			return c.JSON(http.StatusBadRequest, xerrors.Errorf("invalid parameter 'backward': %s", backwardFlightDateParam).Error())
		}
	}

	showBanned := strings.EqualFold(c.QueryParam("show_banned"), "true")
	debug := strings.EqualFold(c.QueryParam("debug"), "true")
	nationalVersion := c.QueryParam("national_version")
	daysCountParam := c.QueryParam("days")
	daysCount := 2
	if len(daysCountParam) > 0 {
		value, err := strconv.Atoi(daysCountParam)
		if err != nil {
			return c.JSON(http.StatusBadRequest, xerrors.Errorf("invalid parameter 'days': %w", err).Error())
		}
		if value < 0 {
			return c.JSON(http.StatusBadRequest, xerrors.Errorf("invalid parameter 'days': %s", daysCountParam).Error())
		}
		daysCount = value
	}

	forwardFightDate := flightp2p.TimeParam{
		Value:   forwardFlightDateParam.ToTime(time.UTC),
		IsLocal: true,
	}
	forwardValue, err := service.GetFlightsP2PByDate(from, to, forwardFightDate, daysCount, nationalVersion, showBanned, debug)

	if err != nil {
		if httpErr, ok := err.(*utils.ErrorWithHTTPCode); ok {
			return c.JSON(httpErr.HTTPCode, fmt.Sprintf("%v", httpErr))
		}
		return c.JSON(http.StatusInternalServerError, fmt.Sprintf("%v", err))
	}

	if len(forwardValue.Dates) == 0 || len(backwardFlightDateParam) == 0 {
		return c.JSON(
			http.StatusOK,
			format.ResponseByDate{
				Forward:  forwardValue,
				Backward: nil,
			},
		)
	}

	backwardFightDate := flightp2p.TimeParam{
		Value:   backwardFlightDateParam.ToTime(time.UTC),
		IsLocal: true,
	}
	backwardValue, err := service.GetFlightsP2PByDate(to, from, backwardFightDate, daysCount, nationalVersion, showBanned, debug)

	if err != nil {
		if httpErr, ok := err.(*utils.ErrorWithHTTPCode); ok {
			return c.JSON(httpErr.HTTPCode, fmt.Sprintf("%v", httpErr))
		}
		return c.JSON(http.StatusInternalServerError, fmt.Sprintf("%v", err))
	}

	if len(backwardValue.Dates) == 0 {
		return c.JSON(
			http.StatusOK,
			format.ResponseByDate{
				Forward:  forwardValue,
				Backward: &backwardValue,
			},
		)
	}

	if flightp2p.AtLeastOneBackwardDateIsEqualOrAfterForwardDate(forwardValue.Dates, backwardValue.Dates) {
		return c.JSON(
			http.StatusOK,
			format.ResponseByDate{
				Forward:  forwardValue,
				Backward: &backwardValue,
			},
		)
	}

	return c.JSON(http.StatusOK, createEmptyResponse(forwardValue))
}

func (h *flightP2PByDateHandler) GetRoute() string {
	return "/flight-p2p-by-date/"
}

func NewFlightP2PByDateHandler(service *storage.Service, config config.P2PConfig) *flightP2PByDateHandler {
	return &flightP2PByDateHandler{
		service:                 service,
		maxP2PStationsInRequest: config.MaxP2PStationsInRequest,
	}
}

// Response that only contains stations
func createEmptyResponse(forwardValue format.DirectDatesAndFlights) format.ResponseByDate {
	value := format.DirectDatesAndFlights{
		DepartureStations: forwardValue.DepartureStations,
		ArrivalStations:   forwardValue.ArrivalStations,
	}
	return format.ResponseByDate{
		Forward:  value,
		Backward: nil,
	}
}

func (h *flightP2PByDateHandler) parseStationsFromParam(
	stationsValuesFromParam []string, direction string) ([]*snapshots.TStationWithCodes, int, error) {
	service := h.service.Instance()
	stationValues := []*snapshots.TStationWithCodes{}
	for _, station := range stationsValuesFromParam {
		if len(station) == 0 {
			return nil, http.StatusBadRequest, xerrors.Errorf("Empty \"%s\" station", direction)
		}
		stationValue, err := lib.ParseStation(station, service.StationProvider())
		if err != nil {
			return nil, http.StatusNotFound, xerrors.Errorf("cannot get \"%s\" station: %w", direction, err)
		}
		tz := service.GetTimeZoneByStationID(int64(stationValue.Station.Id))
		if tz == nil {
			err := xerrors.Errorf("cannot get \"%s\" station (%v) timezone", direction, stationValue.Station.Id)
			return nil, http.StatusInternalServerError, err
		}
		stationValues = append(stationValues, stationValue)
	}
	return stationValues, http.StatusOK, nil
}
