package handlers

import (
	"fmt"
	"net/http"
	"strings"
	"time"

	"github.com/labstack/echo/v4"

	"a.yandex-team.ru/library/go/core/xerrors"
	"a.yandex-team.ru/travel/avia/shared_flights/api/internal/handlers/lib"
	"a.yandex-team.ru/travel/avia/shared_flights/api/internal/services/storage"
	"a.yandex-team.ru/travel/avia/shared_flights/api/internal/storage/timezone"
	"a.yandex-team.ru/travel/avia/shared_flights/api/internal/utils"
	"a.yandex-team.ru/travel/avia/shared_flights/lib/go/dtutil"
	"a.yandex-team.ru/travel/proto/shared_flights/snapshots"
)

type flightP2PScheduleHandler struct {
	service *storage.Service
}

func (h *flightP2PScheduleHandler) Handle(c echo.Context) error {
	service := h.service.Instance()

	form := FlightP2PScheduleForm{}
	if err := form.Fill(c, service, false); err.Error != nil {
		return c.JSON(err.HTTPCode, err.Error.Error())
	}

	if err := form.Validate(); err != nil {
		return c.JSON(http.StatusBadRequest, err.Error())
	}

	value, storageErr := service.GetFlightsP2PSchedule(
		form.departFrom, form.arriveTo, form.nationalVersion, form.showBanned, dtutil.DateToInt(time.Now()))

	if storageErr != nil {
		if httpErr, ok := storageErr.(*utils.ErrorWithHTTPCode); ok {
			return c.JSON(httpErr.HTTPCode, fmt.Sprintf("%v", httpErr))
		}
		return c.JSON(http.StatusInternalServerError, fmt.Sprintf("%v", storageErr))
	}

	return c.JSON(http.StatusOK, value)
}

func (h *flightP2PScheduleHandler) GetRoute() string {
	return "/flight-p2p-schedule/"
}

func NewFlightP2PScheduleHandler(service *storage.Service) *flightP2PScheduleHandler {
	return &flightP2PScheduleHandler{
		service: service,
	}
}

type FlightP2PScheduleForm struct {
	departFrom      []*snapshots.TStationWithCodes
	arriveTo        []*snapshots.TStationWithCodes
	nationalVersion string
	showBanned      bool
	debug           bool
}

func (f *FlightP2PScheduleForm) Fill(
	c echo.Context,
	service *storage.ServiceInstance,
	ignoreDifferentTimezones bool,
) lib.ErrorWithCode {

	params := c.QueryParams()
	fromParam, ok := params["from"]
	if !ok || len(fromParam) == 0 {
		return lib.ErrorWithCode{
			HTTPCode: http.StatusBadRequest,
			Error:    xerrors.New("Missing \"from\" station"),
		}
	}
	toParam, ok := params["to"]
	if !ok || len(toParam) == 0 {
		return lib.ErrorWithCode{
			HTTPCode: http.StatusBadRequest,
			Error:    xerrors.New("Missing \"to\" station"),
		}
	}

	_, debug := params["debug"]

	now := time.Now()
	inAMonth := now.Add(24 * time.Hour * 30)

	from := []*snapshots.TStationWithCodes{}
	var fromTz *time.Location
	fromTzStationID := int32(0)
	for _, station := range fromParam {
		if len(station) == 0 {
			return lib.ErrorWithCode{
				HTTPCode: http.StatusBadRequest,
				Error:    xerrors.New("empty \"from\" station"),
			}
		}
		stationFrom, err := lib.ParseStation(station, service.StationProvider())
		if err != nil {
			return lib.ErrorWithCode{
				HTTPCode: http.StatusNotFound,
				Error:    xerrors.Errorf("cannot get \"from\" station: %w", err),
			}
		}
		tz := service.GetTimeZoneByStationID(int64(stationFrom.Station.Id))
		if tz == nil {
			return lib.ErrorWithCode{
				HTTPCode: http.StatusInternalServerError,
				Error:    xerrors.Errorf("cannot get station (%v) timezone", stationFrom.Station.Id),
			}
		}
		if fromTz == nil {
			fromTz = tz
			fromTzStationID = stationFrom.Station.Id
		} else if !timezone.CompareOffsets(fromTz, tz, now) || !timezone.CompareOffsets(fromTz, tz, inAMonth) {
			if !ignoreDifferentTimezones {
				return lib.ErrorWithCode{
					HTTPCode: http.StatusBadRequest,
					Error: xerrors.Errorf(
						"origin stations %v and %v have different timezones (%v and %v)",
						stationFrom.Station.Id,
						fromTzStationID,
						tz.String(),
						fromTz.String(),
					),
				}
			}
		}
		from = append(from, stationFrom)
	}

	to := []*snapshots.TStationWithCodes{}
	var toTz *time.Location
	toTzStationID := int32(0)
	for _, station := range toParam {
		if len(station) == 0 {
			return lib.ErrorWithCode{
				HTTPCode: http.StatusBadRequest,
				Error:    xerrors.New("empty \"to\" station"),
			}
		}
		stationTo, err := lib.ParseStation(station, service.StationProvider())
		if err != nil {
			return lib.ErrorWithCode{
				HTTPCode: http.StatusNotFound,
				Error:    xerrors.Errorf("cannot get \"to\" station: %w", err),
			}
		}
		tz := service.GetTimeZoneByStationID(int64(stationTo.Station.Id))
		if tz == nil {
			return lib.ErrorWithCode{
				HTTPCode: http.StatusInternalServerError,
				Error:    xerrors.Errorf("cannot get station (%v) timezone", stationTo.Station.Id),
			}
		}
		if toTz == nil {
			toTz = tz
			toTzStationID = stationTo.Station.Id
		} else if !timezone.CompareOffsets(toTz, tz, now) || !timezone.CompareOffsets(toTz, tz, inAMonth) {
			if !ignoreDifferentTimezones {
				return lib.ErrorWithCode{
					HTTPCode: http.StatusBadRequest,
					Error: xerrors.Errorf(
						"destination stations %v and %v have different timezones (%v and %v)",
						stationTo.Station.Id,
						toTzStationID,
						tz.String(),
						toTz.String(),
					),
				}
			}
		}
		to = append(to, stationTo)
	}

	showBanned := strings.EqualFold(c.QueryParam("show_banned"), "true")
	nationalVersion := c.QueryParam("national_version")

	f.departFrom = from
	f.arriveTo = to
	f.nationalVersion = nationalVersion
	f.showBanned = showBanned
	f.debug = debug

	return lib.ErrorWithCode{
		HTTPCode: http.StatusOK,
		Error:    nil,
	}
}

func (f *FlightP2PScheduleForm) Validate() error {
	if len(f.departFrom) == 0 {
		return xerrors.New("provide at least one departure point")
	}
	if len(f.arriveTo) == 0 {
		return xerrors.New("provide at least one arrival point")
	}
	return nil
}
