package handlers

import (
	"fmt"
	"net/http"
	"strconv"
	"strings"
	"time"

	"github.com/labstack/echo/v4"

	"a.yandex-team.ru/library/go/core/xerrors"
	"a.yandex-team.ru/travel/avia/shared_flights/api/internal/services/storage"
	"a.yandex-team.ru/travel/avia/shared_flights/api/internal/services/storage/flight"
	"a.yandex-team.ru/travel/avia/shared_flights/api/internal/utils"
	dir "a.yandex-team.ru/travel/avia/shared_flights/lib/go/direction"
	"a.yandex-team.ru/travel/avia/shared_flights/lib/go/dtutil"
	"a.yandex-team.ru/travel/avia/shared_flights/lib/go/httputil"
	"a.yandex-team.ru/travel/avia/shared_flights/lib/go/strutil"
)

type FlightRangeHanlder struct {
	service *storage.Service
}

type FlightRangeForm struct {
	Carrier         string
	FlightNumber    string
	ReferenceTime   time.Time
	PseudoNow       time.Time
	LimitBefore     int
	LimitAfter      int
	Direction       dir.Direction
	NationalVersion string
	ShowBanned      bool
	TextFormat      bool
}

type flightRangeFormParser interface {
	FillForm(c echo.Context, requireCarrier bool) (form FlightRangeForm, formErr error)
}

type flightRangeFormParserImpl struct{}

func (p *flightRangeFormParserImpl) FillForm(c echo.Context, requireCarrier bool) (form FlightRangeForm, formErr error) {
	if requireCarrier {
		form.Carrier = httputil.Unescape(c.Param("carrier"))
	}
	form.FlightNumber = httputil.Unescape(strutil.RemoveLeadingZeroes(c.Param("flight")))

	referenceDateTime, err := dtutil.DateTimeStringToTimeUTC(c.QueryParam("time_utc"))
	if err != nil {
		formErr = &utils.ErrorWithHTTPCode{
			HTTPCode:     http.StatusBadRequest,
			ErrorMessage: err.Error(),
		}
		return
	}
	if referenceDateTime.IsZero() {
		referenceDateTime = time.Now().UTC()
	}
	form.ReferenceTime = referenceDateTime

	pseudoNow, err := dtutil.DateTimeStringToTimeUTC(c.QueryParam("now"))
	if err != nil {
		formErr = &utils.ErrorWithHTTPCode{
			HTTPCode:     http.StatusBadRequest,
			ErrorMessage: err.Error(),
		}
		return
	}
	if pseudoNow.IsZero() {
		pseudoNow = time.Now()
	}
	form.PseudoNow = pseudoNow

	departureDayPeriodText := c.QueryParam("departure_day_period")
	arrivalDayPeriodText := c.QueryParam("arrival_day_period")

	var periodString string
	var direction dir.Direction

	if len(departureDayPeriodText) > 0 {
		periodString = departureDayPeriodText
		direction = dir.DEPARTURE
	} else if len(arrivalDayPeriodText) > 0 {
		periodString = arrivalDayPeriodText
		direction = dir.ARRIVAL
	} else {
		formErr = &utils.ErrorWithHTTPCode{
			HTTPCode:     http.StatusBadRequest,
			ErrorMessage: "You should define either departure_day_period or arrival_day_period parameter",
		}
		return
	}
	form.Direction = direction

	periodList := strings.Split(periodString, ":")
	if len(periodList) != 2 {
		formErr = &utils.ErrorWithHTTPCode{
			HTTPCode:     http.StatusBadRequest,
			ErrorMessage: "incorrect period format",
		}
		return
	}

	limitBefore, err := strconv.Atoi(periodList[0])
	if err != nil {
		formErr = &utils.ErrorWithHTTPCode{
			HTTPCode:     http.StatusBadRequest,
			ErrorMessage: xerrors.Errorf("incorrect period-before format: %+v", err).Error(),
		}
		return
	}
	if limitBefore < 0 {
		formErr = &utils.ErrorWithHTTPCode{
			HTTPCode:     http.StatusBadRequest,
			ErrorMessage: xerrors.Errorf("incorrect period-before value: %+v", limitBefore).Error(),
		}
		return
	}
	form.LimitBefore = limitBefore

	limitAfter, err := strconv.Atoi(periodList[1])
	if err != nil {
		formErr = &utils.ErrorWithHTTPCode{
			HTTPCode:     http.StatusBadRequest,
			ErrorMessage: xerrors.Errorf("incorrect period-after format: %+v", err).Error(),
		}
		return
	}
	if limitAfter < 0 {
		formErr = &utils.ErrorWithHTTPCode{
			HTTPCode:     http.StatusBadRequest,
			ErrorMessage: xerrors.Errorf("incorrect period-after value: %+v", limitAfter).Error(),
		}
		return
	}
	form.LimitAfter = limitAfter

	form.ShowBanned = strings.EqualFold(c.QueryParam("show_banned"), "true")
	form.NationalVersion = c.QueryParam("national_version")
	form.TextFormat = strings.EqualFold(c.QueryParam("text"), "true")
	return
}

func newFlightRangeFormParser() flightRangeFormParser {
	return &flightRangeFormParserImpl{}
}

func (h *FlightRangeHanlder) Handle(c echo.Context) error {
	form, formErr := newFlightRangeFormParser().FillForm(c, true)
	if formErr != nil {
		if httpErr, ok := formErr.(*utils.ErrorWithHTTPCode); ok {
			return c.JSON(httpErr.HTTPCode, fmt.Sprintf("%v", httpErr))
		}
		return c.JSON(http.StatusInternalServerError, fmt.Sprintf("%v", formErr))
	}

	value, storageErr := h.service.Instance().GetFlightRange(
		flight.NewCarrierParamByText(form.Carrier),
		form.FlightNumber,
		form.NationalVersion,
		form.ShowBanned,
		form.PseudoNow,
		form.LimitBefore,
		form.LimitAfter,
		form.Direction,
		form.ReferenceTime,
	)

	if storageErr != nil {
		if httpErr, ok := storageErr.(*utils.ErrorWithHTTPCode); ok {
			return c.JSON(httpErr.HTTPCode, fmt.Sprintf("%v", httpErr))
		}
		return c.JSON(http.StatusInternalServerError, fmt.Sprintf("%v", storageErr))
	}

	if form.TextFormat {
		return c.String(http.StatusOK, ConvertFlightsToText(value))
	}

	return c.JSON(http.StatusOK, value)
}

func (h *FlightRangeHanlder) GetRoute() string {
	return "/flight-range/:carrier/:flight/"
}

func NewFlightRangeHandler(service *storage.Service) *FlightRangeHanlder {
	return &FlightRangeHanlder{
		service: service,
	}
}
