package handlers

import (
	"fmt"
	"net/http"
	"strings"
	"time"

	"github.com/labstack/echo/v4"

	"a.yandex-team.ru/travel/avia/shared_flights/api/internal/services/storage"
	p2pformat "a.yandex-team.ru/travel/avia/shared_flights/api/internal/services/storage/flight_p2p/format"
	"a.yandex-team.ru/travel/avia/shared_flights/api/internal/services/storage/flight_schedule/format"
	"a.yandex-team.ru/travel/avia/shared_flights/api/internal/utils"
	"a.yandex-team.ru/travel/avia/shared_flights/lib/go/dtutil"
	"a.yandex-team.ru/travel/avia/shared_flights/lib/go/httputil"
	"a.yandex-team.ru/travel/avia/shared_flights/lib/go/strutil"
)

type flightScheduleHandler struct {
	service *storage.Service
}

type flightScheduleForm struct {
	carrier         string
	flightNumber    string
	nationalVersion string
	showBanned      bool
	textFormat      bool
}

func (h *flightScheduleHandler) Handle(c echo.Context) error {

	form := h.fillForm(c)

	value, storageErr := h.service.Instance().GetFlightSchedule(
		form.carrier,
		form.flightNumber,
		form.nationalVersion,
		form.showBanned,
		dtutil.DateToInt(time.Now().AddDate(0, 0, -31)),
	)

	if storageErr != nil {
		if httpErr, ok := storageErr.(*utils.ErrorWithHTTPCode); ok {
			return c.JSON(httpErr.HTTPCode, fmt.Sprintf("%v", httpErr))
		}
		return c.JSON(http.StatusInternalServerError, fmt.Sprintf("%v", storageErr))
	}

	if form.textFormat {
		return c.String(http.StatusOK, ConvertSchedulesToText(value))
	}

	return c.JSON(http.StatusOK, value)
}

func ConvertSchedulesToText(value format.Response) string {
	lines := make([]string, 0, len(value.Schedules))

	lines = append(lines, fmt.Sprintf("title: %v", value.Title))
	lines = append(lines, fmt.Sprintf("carrier ID: %v", value.CarrierID))

	for _, schedule := range value.Schedules {
		if len(schedule.Masks) == 0 {
			continue
		}
		for i := 0; i < len(schedule.Masks)-1; i++ {
			if i < len(schedule.Masks)-1 {
				lines = append(lines, getMaskText(schedule.Masks[i]))
			}
		}
		line := strings.Join(
			[]string{
				getMaskText(schedule.Masks[len(schedule.Masks)-1]),
				getRouteText(schedule.Route),
			},
			" ",
		)
		lines = append(lines, line)
	}

	return strings.Join(lines, "\n")
}

func getMaskText(mask p2pformat.Mask) string {
	return strings.Join(
		[]string{
			fmt.Sprintf("%v", mask.From),
			fmt.Sprintf("%v", mask.Until),
			dtutil.OperatingDays(mask.On).ReadableString(),
		},
		" ",
	)
}

func getRouteText(route []format.StopPoint) string {
	parts := make([]string, 0, len(route))
	for _, stopPoint := range route {
		if stopPoint.ArrivalTime != "" {
			parts = append(parts, []string{
				"-",
				stopPoint.AirportCode,
				stopPoint.ArrivalTime,
				overnightToString(int32(stopPoint.ArrivalDayShift)),
			}...)
		}
		if stopPoint.DepartureTime != "" {
			parts = append(parts, []string{
				"|",
				stopPoint.AirportCode,
				stopPoint.DepartureTime,
				overnightToString(int32(stopPoint.DepartureDayShift)),
			}...)
		}
	}
	return strings.Join(parts, " ")
}

func (h *flightScheduleHandler) fillForm(c echo.Context) (form flightScheduleForm) {
	form.carrier = httputil.Unescape(c.Param("carrier"))
	form.flightNumber = httputil.Unescape(strutil.RemoveLeadingZeroes(c.Param("flight")))
	form.nationalVersion = c.QueryParam("national_version")
	form.showBanned = strings.EqualFold(c.QueryParam("show_banned"), "true")
	form.textFormat = strings.EqualFold(c.QueryParam("text"), "true")

	return
}

func (h *flightScheduleHandler) GetRoute() string {
	return "/flight-schedule/:carrier/:flight/"
}

func NewFlightScheduleHandler(service *storage.Service) *flightScheduleHandler {
	return &flightScheduleHandler{
		service: service,
	}
}
