package handlers

import (
	"net/http"
	"strings"

	"github.com/labstack/echo/v4"

	"a.yandex-team.ru/library/go/core/xerrors"
	"a.yandex-team.ru/travel/avia/shared_flights/api/internal/handlers/lib"
	"a.yandex-team.ru/travel/avia/shared_flights/api/internal/services/storage"
)

type delayedFlightsHandler struct {
	service *storage.Service
}

func (h *delayedFlightsHandler) Handle(c echo.Context) error {
	service := h.service.Instance()

	params := c.QueryParams()
	allStations := strings.EqualFold(c.QueryParam("all_stations"), "true")
	stationParam, ok := params["station"]
	if !allStations && (!ok || len(stationParam) == 0) {
		return c.JSON(http.StatusBadRequest, "Missing \"station\" parameter")
	}

	stations := []int64{}
	if !allStations {
		for _, stationText := range stationParam {
			if len(stationText) == 0 {
				return c.JSON(http.StatusBadRequest, "Empty station value is not allowed")
			}
			station, err := lib.ParseStation(stationText, service.StationProvider())
			if err != nil {
				return c.JSON(http.StatusNotFound, xerrors.Errorf("cannot recognize station: %w", err).Error())
			}
			stations = append(stations, int64(station.Station.Id))
		}
	}

	debug := strings.EqualFold(c.QueryParam("debug"), "true")

	value := service.GetDelayedFlights(stations, debug, allStations)

	return c.JSON(http.StatusOK, value)
}

func (h *delayedFlightsHandler) GetRoute() string {
	return "/delayed-flights/"
}

func NewDelayedFlightsHandler(service *storage.Service) *delayedFlightsHandler {
	return &delayedFlightsHandler{
		service: service,
	}
}
