package handlers

import (
	"fmt"
	"net/http"
	"strings"
	"time"

	"github.com/labstack/echo/v4"

	"a.yandex-team.ru/library/go/core/xerrors"
	"a.yandex-team.ru/travel/avia/shared_flights/api/internal/handlers/lib"
	"a.yandex-team.ru/travel/avia/shared_flights/api/internal/services/storage"
	"a.yandex-team.ru/travel/avia/shared_flights/api/internal/utils"
	dir "a.yandex-team.ru/travel/avia/shared_flights/lib/go/direction"
	"a.yandex-team.ru/travel/avia/shared_flights/lib/go/httputil"
	"a.yandex-team.ru/travel/proto/shared_flights/snapshots"
)

type stationScheduleHandler struct {
	service *storage.Service
}

type stationScheduleForm struct {
	station         *snapshots.TStationWithCodes
	direction       dir.Direction
	nationalVersion string
	formErr         error
}

func (h *stationScheduleHandler) Handle(c echo.Context) error {

	form, formErr := h.fillForm(c)
	if formErr != nil {
		if httpErr, ok := formErr.(*utils.ErrorWithHTTPCode); ok {
			return c.JSON(httpErr.HTTPCode, fmt.Sprintf("%v", httpErr))
		}
		return c.JSON(http.StatusInternalServerError, fmt.Sprintf("%v", formErr))
	}

	now := time.Now().UTC()

	value, storageErr := h.service.Instance().GetStationSchedule(form.station, form.direction, form.nationalVersion, now)

	if storageErr != nil {
		if httpErr, ok := storageErr.(*utils.ErrorWithHTTPCode); ok {
			return c.JSON(httpErr.HTTPCode, fmt.Sprintf("%v", httpErr))
		}
		return c.JSON(http.StatusInternalServerError, fmt.Sprintf("%v", storageErr))
	}

	return c.JSON(http.StatusOK, value)
}

func (h *stationScheduleHandler) fillForm(c echo.Context) (
	form stationScheduleForm,
	formErr error,
) {
	var err error
	stationParam := httputil.Unescape(c.Param("station"))
	if len(stationParam) == 0 {
		formErr = &utils.ErrorWithHTTPCode{
			HTTPCode:     http.StatusBadRequest,
			ErrorMessage: xerrors.Errorf("missing \"station\" parameter").Error(),
		}
		return
	}
	form.station, err = lib.ParseStation(stationParam, h.service.Instance().StationProvider())
	if err != nil {
		formErr = &utils.ErrorWithHTTPCode{
			HTTPCode:     http.StatusNotFound,
			ErrorMessage: xerrors.Errorf("cannot get station: %w", err).Error(),
		}
		return
	}
	directionString := strings.ToLower(c.QueryParam("direction"))
	form.direction = dir.DEPARTURE
	if len(directionString) > 0 {
		if form.direction, err = dir.FromString(directionString); err != nil {
			formErr = &utils.ErrorWithHTTPCode{
				HTTPCode:     http.StatusBadRequest,
				ErrorMessage: xerrors.Errorf("invalid parameter 'direction': %w", err).Error(),
			}
			return
		}
	}

	form.nationalVersion = c.QueryParam("national_version")

	return
}

func (h *stationScheduleHandler) GetRoute() string {
	return "/flight-board/:station/schedule/"
}

// NewFlightBoardScheduleHandler constructor for station schedule requests handler
func NewFlightBoardScheduleHandler(service *storage.Service) *stationScheduleHandler {
	return &stationScheduleHandler{
		service: service,
	}
}
