package server

import (
	"runtime"
	"time"

	log "a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/core/xerrors"
	aviaMetrics "a.yandex-team.ru/travel/avia/library/go/metrics"
	"a.yandex-team.ru/travel/avia/shared_flights/api/internal/config"
	"a.yandex-team.ru/travel/avia/shared_flights/api/internal/metrics"
	"a.yandex-team.ru/travel/avia/shared_flights/api/internal/services/loadsnapshot"
	"a.yandex-team.ru/travel/avia/shared_flights/lib/go/logger"
)

// Keeps reloading the data into storage
type loadSnapshotWorker struct {
	SleepBetweenReloads time.Duration
	MaxSuccessiveErrors int
}

func NewLoadSnapshotWorker(config config.SnapshotWorkerConfig) *loadSnapshotWorker {
	instance := loadSnapshotWorker(config)
	return &instance
}

func (w *loadSnapshotWorker) RunWorker(loadSnapshotService loadsnapshot.Service) error {
	var CurrentSuccessiveErrors = 0
	stopMetrics := false
	defer func() {
		stopMetrics = true
		if err := recover(); err != nil {
			logger.Logger().Error("Snapshot worker error", log.Reflect("error", err))
		}
	}()

	lastBaseLoadTime := time.Time{}
	lastDeltaLoadTime := time.Time{}

	go func() {
		t := time.NewTicker(time.Second)
		defer t.Stop()
		for range t.C {
			if !lastBaseLoadTime.IsZero() {
				aviaMetrics.GlobalMetrics().
					Timer(metrics.LoaderPrefix, metrics.BaseTimeSinceLastLoad).
					RecordDuration(time.Since(lastBaseLoadTime))
			}
			if !lastDeltaLoadTime.IsZero() {
				aviaMetrics.GlobalMetrics().
					Timer(metrics.LoaderPrefix, metrics.DeltaTimeSinceLastLoad).
					RecordDuration(time.Since(lastDeltaLoadTime))
			}
			if stopMetrics {
				return
			}
		}
	}()

	currentBase := ""
	currentDelta := ""
	runtime.LockOSThread()
	defer runtime.UnlockOSThread()
	for {
		result, err := loadSnapshotService.LoadSnapshot(currentBase, currentDelta, false)

		if err != nil {
			CurrentSuccessiveErrors++
			logger.Logger().Error(
				"Unable to load snapshot",
				log.Error(err),
				log.Int("successive errors count", CurrentSuccessiveErrors),
			)
			if CurrentSuccessiveErrors >= w.MaxSuccessiveErrors {
				return xerrors.Errorf("max successive loader snapshot count exceeded: %w", err)
			}
			time.Sleep(w.SleepBetweenReloads)
			continue
		}
		CurrentSuccessiveErrors = 0

		logger.Logger().Info("Current snapshot: ", log.Reflect("result", result))
		if currentBase != result.BaseMarker {
			lastBaseLoadTime = time.Now()
		}
		currentBase = result.BaseMarker

		if currentDelta != result.DeltaMarker {
			lastDeltaLoadTime = time.Now()
		}
		currentDelta = result.DeltaMarker

		time.Sleep(w.SleepBetweenReloads)
	}
}
