package server

import (
	"time"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/core/xerrors"
	aviaMetrics "a.yandex-team.ru/travel/avia/library/go/metrics"
	"a.yandex-team.ru/travel/avia/shared_flights/api/internal/appconst"
	"a.yandex-team.ru/travel/avia/shared_flights/api/internal/metrics"
	"a.yandex-team.ru/travel/avia/shared_flights/api/internal/services/storage"
	"a.yandex-team.ru/travel/avia/shared_flights/lib/go/direction"
	"a.yandex-team.ru/travel/avia/shared_flights/lib/go/logger"
)

const noFilter = ""

func RunMetricsWorker(service *storage.Service) {
	go departuresWorker(service)
	go storageAvailabilityWorker(service)
}

func storageAvailabilityWorker(service *storage.Service) {
	for range time.Tick(time.Second) {
		available := 0.
		if service.Instance().Storage().IsAvailable {
			available = 1.
		}
		aviaMetrics.GlobalMetrics().Gauge(metrics.LoaderPrefix, metrics.StorageOk).Set(available)
	}
}

func departuresWorker(service *storage.Service) {
	for {
		if service.Instance().Storage().IsAvailable {
			break
		}
		time.Sleep(time.Second)
	}
	for range time.Tick(time.Minute) {
		if err := fetchAirportDeparturesWithStatuses(
			service, "SVO", metrics.DeparturesStatusesSVO); err != nil {
			continue
		}
		time.Sleep(time.Second)
		if err := fetchAirportDeparturesWithStatuses(
			service, "DME", metrics.DeparturesStatusesDME); err != nil {
			continue
		}
		time.Sleep(time.Second)
		if err := fetchAirportDeparturesWithStatuses(
			service, "VKO", metrics.DeparturesStatusesVKO); err != nil {
			continue
		}
		time.Sleep(time.Second)
		if err := fetchAirportDeparturesWithStatuses(
			service, "LED", metrics.DeparturesStatusesLED); err != nil {
			continue
		}
		time.Sleep(time.Second)

		if err := fetchAirportDeparturesTomorrow(
			service, "SVO", metrics.DeparturesSVO); err != nil {
			continue
		}
		time.Sleep(time.Second)
		if err := fetchAirportDeparturesTomorrow(
			service, "DME", metrics.DeparturesDME); err != nil {
			continue
		}
		time.Sleep(time.Second)
		if err := fetchAirportDeparturesTomorrow(
			service, "VKO", metrics.DeparturesVKO); err != nil {
			continue
		}
		time.Sleep(time.Second)
		if err := fetchAirportDeparturesTomorrow(
			service, "LED", metrics.DeparturesLED); err != nil {
			continue
		}
		time.Sleep(time.Second)
		if err := fetchAirportDeparturesTomorrow(
			service, "SVX", metrics.DeparturesSVX); err != nil {
			continue
		}
		time.Sleep(time.Second)
		if err := fetchAirportDeparturesTomorrow(
			service, "ATL", metrics.DeparturesATL); err != nil {
			continue
		}
	}
}

func fetchAirportDeparturesWithStatuses(
	service *storage.Service, airportCode string, statusesMetricName aviaMetrics.MetricName) error {
	aprt, ok := service.Instance().StationProvider().ByIata(airportCode)
	if !ok || aprt == nil {
		logger.Logger().Error("Cannot get flight board. No such airport", log.String("code", airportCode))
		return xerrors.Errorf("cannot get flight board: no such airport: %v", airportCode)
	}
	response, err := service.Instance().GetFlightBoard(
		aprt,
		time.Now().Add(-1*time.Hour),
		time.Now().Add(24*time.Hour),
		0,
		true,
		direction.DEPARTURE,
		"",
		"",
		noFilter,
		time.Now(),
	)
	if err != nil {
		logger.Logger().Error("Cannot get flight board", log.Error(err))
		return xerrors.Errorf("cannot get flight board: %w", err)
	}
	departuresWithStatuses := 0
	for _, item := range response.Flights {
		if item.Status.DepartureStatus != string(appconst.FlightStatusUnknown) {
			departuresWithStatuses++
		}
	}
	aviaMetrics.GlobalMetrics().Gauge(metrics.StorageState, statusesMetricName).Set(float64(departuresWithStatuses))
	return nil
}

func fetchAirportDeparturesTomorrow(
	service *storage.Service, airportCode string, departuresMetricName aviaMetrics.MetricName) error {
	aprt, ok := service.Instance().StationProvider().ByIata(airportCode)
	if !ok || aprt == nil {
		logger.Logger().Error("Cannot get flight board. No such airport", log.String("code", airportCode))
		return xerrors.Errorf("cannot get flight board: no such airport: %v", airportCode)
	}
	response, err := service.Instance().GetFlightBoard(
		aprt,
		time.Now().Add(24*time.Hour),
		time.Now().Add(48*time.Hour),
		0,
		true,
		direction.DEPARTURE,
		"",
		"",
		noFilter,
		time.Now(),
	)
	if err != nil {
		logger.Logger().Error("Cannot get flight board", log.Error(err))
		return xerrors.Errorf("cannot get flight board: %w", err)
	}
	aviaMetrics.GlobalMetrics().Gauge(metrics.StorageState, departuresMetricName).Set(float64(len(response.Flights)))
	return nil
}
