package server

import (
	"fmt"
	"net/http"

	"github.com/labstack/echo/v4"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/core/xerrors"
	"a.yandex-team.ru/travel/avia/shared_flights/api/internal/services/storage"
	"a.yandex-team.ru/travel/avia/shared_flights/lib/go/logger"
)

func StorageServiceAvailable(service *storage.Service) echo.MiddlewareFunc {
	middlewareFunc := func(handlerFunc echo.HandlerFunc) echo.HandlerFunc {
		return func(c echo.Context) error {
			if err := service.Available(); err != nil {
				return c.JSON(http.StatusServiceUnavailable, err.Error())
			}
			return handlerFunc(c)
		}
	}
	return middlewareFunc
}

func PanicLoggerMiddleware(handlerFunc echo.HandlerFunc) echo.HandlerFunc {
	return func(c echo.Context) (err error) {
		defer func() {
			if e := recover(); e != nil {
				logger.Logger().Error(
					"Uncaught error",
					log.Reflect("error", e),
					log.String("verbose", fmt.Sprintf("%+v", e)),
					log.String("struct", fmt.Sprintf("%#v", e)),
				)
				switch t := e.(type) {
				case fmt.Stringer:
					err = xerrors.Errorf("Server got itself into trouble: %s", t.String())
				case string:
					err = xerrors.Errorf("Server got itself into trouble: %v", t)
				case error:
					err = xerrors.Errorf("Server got itself into trouble: %w", t)
				default:
					err = xerrors.Errorf("Server got itself into trouble: %+v %#v", t, t)
				}
			}
		}()
		err = handlerFunc(c)
		return err
	}
}
