package server

import (
	"fmt"
	"time"

	"github.com/labstack/echo/v4"
	"github.com/labstack/echo/v4/middleware"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/travel/avia/shared_flights/api/internal/config"
	"a.yandex-team.ru/travel/avia/shared_flights/api/internal/handlers"
	loadSnapshotServices "a.yandex-team.ru/travel/avia/shared_flights/api/internal/services/loadsnapshot"
	"a.yandex-team.ru/travel/avia/shared_flights/api/internal/services/storage"
	"a.yandex-team.ru/travel/avia/shared_flights/lib/go/logger"
)

type server struct {
}

func NewServer() *server {
	instance := server{}
	return &instance
}

func (s *server) RunServer(
	storageService *storage.Service,
	loadSnapshotService loadSnapshotServices.Service,
	config *config.Config,
	apiPort int) {

	logger.Logger().Info("Starting shared flights api instance")

	echoServer := echo.New()
	echoServer.Pre(middleware.AddTrailingSlash())

	pingHandler := handlers.NewPingHandler(storageService)
	echoServer.GET(pingHandler.GetRoute(), pingHandler.Handle)
	echoServer.POST(pingHandler.GetShutdownRoute(), pingHandler.HandleShutdown)

	storageAccessorsHandler := handlers.NewStorageAccessorHandler(storageService)
	echoServer.GET(storageAccessorsHandler.GetCarrierRoute(), storageAccessorsHandler.GetCarrier)
	echoServer.GET(storageAccessorsHandler.GetFlightBaseRoute(), storageAccessorsHandler.GetFlightBase)
	echoServer.GET(storageAccessorsHandler.GetDopFlightBaseRoute(), storageAccessorsHandler.GetDopFlightBase)
	echoServer.GET(storageAccessorsHandler.GetFlightPatternRoute(), storageAccessorsHandler.GetFlightPattern)
	echoServer.GET(storageAccessorsHandler.GetFlightStatusRoute(), storageAccessorsHandler.GetFlightStatus)
	echoServer.GET(storageAccessorsHandler.GetRawFlightRoute(), storageAccessorsHandler.GetRawFlight)
	echoServer.GET(storageAccessorsHandler.GetRawCodeshareFlightsRoute(), storageAccessorsHandler.GetRawCodeshareFlights)
	echoServer.GET(storageAccessorsHandler.GetRawP2PFlightsRoute(), storageAccessorsHandler.GetRawP2PFlights)
	echoServer.GET(storageAccessorsHandler.GetIataCorrectionRulesRoute(), storageAccessorsHandler.GetIataCorrectionRules)
	echoServer.GET(storageAccessorsHandler.GetFlightStatusSourcesRoute(), storageAccessorsHandler.GetFlightStatusSources)
	echoServer.GET(storageAccessorsHandler.GetFlightMergeRulesRoute(), storageAccessorsHandler.GetFlightMergeRules)
	echoServer.GET(storageAccessorsHandler.GetPopularityScoresRoute(), storageAccessorsHandler.GetPopularityScores)
	echoServer.GET(storageAccessorsHandler.GetPopularCarriersForFlightRoute(), storageAccessorsHandler.GetPopularCarriersForFlight)

	flightHandler := handlers.NewFlightHandler(storageService)
	echoServer.GET(flightHandler.GetRoute(), flightHandler.Handle)

	flightListHandler := handlers.NewFlightListHandler(storageService, config.APIConfig)
	echoServer.GET(flightListHandler.GetRoute(), flightListHandler.Handle)
	echoServer.POST(flightListHandler.GetRoute(), flightListHandler.Handle)

	flightP2PHandler := handlers.NewFlightP2PHandler(storageService, config.APIConfig.P2P)
	echoServer.GET(flightP2PHandler.GetRoute(), flightP2PHandler.Handle)

	flightP2PByDateHandler := handlers.NewFlightP2PByDateHandler(storageService, config.APIConfig.P2P)
	echoServer.GET(flightP2PByDateHandler.GetRoute(), flightP2PByDateHandler.Handle)

	flightP2PScheduleHandler := handlers.NewFlightP2PScheduleHandler(storageService)
	echoServer.GET(flightP2PScheduleHandler.GetRoute(), flightP2PScheduleHandler.Handle)

	flightP2PSummaryHandler := handlers.NewFlightP2PSummaryHandler(storageService)
	echoServer.GET(flightP2PSummaryHandler.GetRoute(), flightP2PSummaryHandler.Handle)

	flightP2PSegmentInfoHandlerr := handlers.NewFlightP2PSegmentInfoHandler(storageService)
	echoServer.GET(flightP2PSegmentInfoHandlerr.GetRoute(), flightP2PSegmentInfoHandlerr.Handle)

	flightRangeHandler := handlers.NewFlightRangeHandler(storageService)
	echoServer.GET(flightRangeHandler.GetRoute(), flightRangeHandler.Handle)

	flightRangeMultiHandler := handlers.NewFlightRangeMultiHandler(storageService)
	echoServer.GET(flightRangeMultiHandler.GetRoute(), flightRangeMultiHandler.Handle)

	flightBoardHandler := handlers.NewFlightBoardHandler(storageService)
	echoServer.GET(flightBoardHandler.GetRoute(), flightBoardHandler.Handle)

	flightScheduleHandler := handlers.NewFlightScheduleHandler(storageService)
	echoServer.GET(flightScheduleHandler.GetRoute(), flightScheduleHandler.Handle)

	flightStationHandler := handlers.NewFlightStationHandler(storageService)
	echoServer.GET(flightStationHandler.GetRoute(), flightStationHandler.Handle)

	flightBoardScheduleHandler := handlers.NewFlightBoardScheduleHandler(storageService)
	echoServer.GET(flightBoardScheduleHandler.GetRoute(), flightBoardScheduleHandler.Handle)

	delayedFlightsHandler := handlers.NewDelayedFlightsHandler(storageService)
	echoServer.GET(delayedFlightsHandler.GetRoute(), delayedFlightsHandler.Handle)

	loadSnapshotHandler := handlers.NewLoadSnapshotHandler(loadSnapshotService)
	echoServer.GET(loadSnapshotHandler.GetRoute(), loadSnapshotHandler.Handle)

	flightNumberToCarrierHandler := handlers.NewFlightNumbersToCarrierHandler(storageService)
	echoServer.POST(flightNumberToCarrierHandler.GetRoute(), flightNumberToCarrierHandler.Handle)
	echoServer.GET(flightNumberToCarrierHandler.GetRoute(), flightNumberToCarrierHandler.Handle)
	echoServer.GET(flightNumberToCarrierHandler.GetAmbiguityRoute(), flightNumberToCarrierHandler.AmbiguityHandler)

	flightNumbersValidator := handlers.NewFlightRangeAccessibleHandler(storageService)
	echoServer.POST(flightNumbersValidator.GetRoute(), flightNumbersValidator.Handle)
	echoServer.GET(flightNumbersValidator.GetRoute(), flightNumbersValidator.Handle)

	flightsNetworkHandler := handlers.SlowQueriesHandler(handlers.NewFlightsNetworkHandler(storageService), config.APIConfig)
	echoServer.GET(flightsNetworkHandler.GetRoute(), flightsNetworkHandler.Handle)

	aeroflotConnectionsHandler := handlers.NewAeroflotConnectionsHandler(storageService)
	echoServer.GET(aeroflotConnectionsHandler.GetRoute(), aeroflotConnectionsHandler.Handle)

	aeroflotConnectingVariantsHandler := handlers.NewAeroflotConnectingVariantsHandler(storageService)
	echoServer.GET(aeroflotConnectingVariantsHandler.GetRoute(), aeroflotConnectingVariantsHandler.Handle)

	echoServer.Use(StorageServiceAvailable(storageService), PanicLoggerMiddleware)

	// Keep loading data snapshots into storage
	go func() {
		for {
			logger.Logger().Info("Starting snapshot loading worker")
			err := NewLoadSnapshotWorker(config.SnapshotWorkerConfig).RunWorker(loadSnapshotService)
			if err != nil {
				logger.Logger().Error("Snapshot worker has stopped with error", log.Error(err))
			} else {
				logger.Logger().Info("Snapshot worker has stopped without error")
			}
			time.Sleep(time.Minute)
		}
	}()
	go RunMetricsWorker(storageService)

	logger.Logger().Fatal("Server error", log.Error(echoServer.Start(fmt.Sprintf(":%d", apiPort))))
}
