package dumper

import (
	"os"
	"path"

	"github.com/golang/protobuf/proto"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/core/xerrors"
	flightp2p "a.yandex-team.ru/travel/avia/shared_flights/api/internal/services/storage/flight_p2p"
	"a.yandex-team.ru/travel/avia/shared_flights/api/internal/services/storage/flight_p2p/format"
	"a.yandex-team.ru/travel/avia/shared_flights/api/internal/services/storage/segment"
	"a.yandex-team.ru/travel/avia/shared_flights/api/pkg/structs"
	"a.yandex-team.ru/travel/avia/shared_flights/lib/go/dtutil"
	"a.yandex-team.ru/travel/avia/shared_flights/lib/go/logger"
	dictsbase "a.yandex-team.ru/travel/library/go/dicts/base"
	"a.yandex-team.ru/travel/proto/dicts/avia"
)

type iFlightStorage interface {
	GetFlightPatterns() map[int32]*structs.FlightPattern
	GetFlightBase(id int32, isDopFlight bool) (flightBase structs.FlightBase, err error)
}

func Dump(flightStorage iFlightStorage) {
	var counters = make(map[interface{}]int)
	var flights = fetchFlights(flightStorage, counters)
	for k, v := range counters {
		logger.Logger().Infof("%T: %v", k, v)
	}
	flightsToProto(flights, "output", "flight_schedule.data")
	logger.Logger().Infof("\n\nThere are %d flights in total\n\n", len(flights))
}

func fetchFlights(flightStorage iFlightStorage, counters map[interface{}]int) []tFlight {
	var err error
	var flights []tFlight
	type countIsCodeshare struct{}

	patternsByFlightNumber := make(map[tFlightNumber][]*structs.FlightPattern)
	for _, v := range flightStorage.GetFlightPatterns() {
		if v.IsCodeshare {
			counters[countIsCodeshare{}] += 1
			continue
		}

		patternsByFlightNumber[tFlightNumber{
			v.MarketingCarrier,
			v.MarketingFlightNumber,
		}] = append(
			patternsByFlightNumber[tFlightNumber{
				v.MarketingCarrier,
				v.MarketingFlightNumber,
			}],
			v,
		)
	}

	logger.Logger().Info("Done making flight pattern map by flight number")
	for flightNumber, patterns := range patternsByFlightNumber {
		var flightSchedules []tSchedule
		flightSchedules, err = singleFlightSchedules(flightStorage, patterns, counters)
		if err != nil {
			continue
		}

		flights = append(flights, tFlight{
			Title:     patterns[0].FlightTitle(),
			AirlineID: flightNumber.carrier,
			Schedules: flightSchedules,
		})
	}
	logger.Logger().Info("Successfully created flight cache")
	return flights
}

func singleFlightSchedules(flightStorage iFlightStorage, patterns []*structs.FlightPattern, counters map[interface{}]int) (tSchedules, error) {
	var err error
	type countDisjointLegSequence struct{}
	type countNoFlightBase struct{}
	type countDisjointRoutePoints struct{}
	type countNoFlightPatternsInGroup struct{}

	if len(patterns) == 0 {
		counters[countNoFlightPatternsInGroup{}] += 1
		return nil, xerrors.New("no flight patterns in group")
	}

	if err = filterDisjointLegSequence(patterns); err != nil {
		counters[countDisjointLegSequence{}] += 1
		return nil, xerrors.Errorf("disjoint leg sequence: %w", err)
	}
	if err = filterWithoutFlightBase(patterns, flightStorage); err != nil {
		counters[countNoFlightBase{}] += 1
		return nil, xerrors.Errorf("no flight base: %w", err)
	}

	if err = filterDisjointRoutePoints(patterns, flightStorage); err != nil {
		counters[countDisjointRoutePoints{}] += 1
		return nil, xerrors.Errorf("disjoint route points: %w", err)
	}

	var flightSchedules tSchedules
	if flightSchedules, err = groupBySchedule(
		createSchedulesForFlight(
			patterns,
			flightStorage,
		),
	); err != nil {
		return nil, xerrors.Errorf("cannot create schedule: %w", err)
	}
	return flightSchedules, nil
}

func flightsToProto(flights []tFlight, directory, filename string) {
	var file *os.File
	var err error
	if err = os.MkdirAll(directory, 0755); err != nil {
		logger.Logger().Error("Unable to create directory", log.Error(err))
	}
	outputPath := path.Join(directory, filename)
	if file, err = os.OpenFile(outputPath, os.O_CREATE|os.O_TRUNC|os.O_WRONLY, 0755); err != nil {
		logger.Logger().Error("Unable to open file", log.Error(err))
		return
	}
	defer func() {
		if closeErr := file.Close(); closeErr != nil {
			logger.Logger().Error("Error closing file", log.Error(closeErr))
		}
	}()
	var protoGenerator *dictsbase.BytesGenerator
	if protoGenerator, err = dictsbase.BuildGeneratorForWriter(file); err != nil {
		logger.Logger().Error(
			"Cannot initialize proto generator for file",
			log.Reflect("file", file),
			log.Error(err),
		)
	}

	for _, flt := range flights {
		var bytes []byte
		if bytes, err = proto.Marshal(flightToProtoMessage(flt)); err != nil {
			logger.Logger().Error(
				"Cannot marshal proto message",
				log.Reflect("flight", flt),
				log.Error(err),
			)
		}
		if err = protoGenerator.Write(bytes); err != nil {
			logger.Logger().Error(
				"Cannot write proto message to file",
				log.Reflect("flight", flt),
				log.Error(err),
			)
		}
	}

}

func flightToProtoMessage(flt tFlight) proto.Message {
	return flt.proto()
}

type tFlightNumber struct {
	carrier int32
	number  string
}

type tRoutePoint segment.RoutePoint
type tRoute segment.Route

func (p tRoutePoint) proto() *avia.TRoutePoint {
	return &avia.TRoutePoint{
		AirportID:         p.AirportID,
		ArrivalTime:       p.ArrivalTime,
		ArrivalDayShift:   int32(p.ArrivalDayShift),
		DepartureTime:     p.DepartureTime,
		DepartureDayShift: int32(p.DepartureDayShift),
	}
}

func (r tRoute) proto() []*avia.TRoutePoint {
	var protoRoute []*avia.TRoutePoint
	for _, routePoint := range r {
		protoRoute = append(protoRoute, tRoutePoint(routePoint).proto())
	}
	return protoRoute
}

type tFormatMasks []format.Mask

func (m tFormatMasks) proto() []*avia.TMask {
	var protoMasks []*avia.TMask
	for _, mask := range m {
		protoMasks = append(protoMasks, &avia.TMask{
			From:  string(mask.From),
			Until: string(mask.Until),
			On:    mask.On,
		})
	}
	return protoMasks
}

type tSchedule struct {
	Route tRoute
	Masks tFormatMasks
}

func (s tSchedule) proto() *avia.TSchedule {
	return &avia.TSchedule{
		Route: s.Route.proto(),
		Masks: s.Masks.proto(),
	}
}

type tSchedules []tSchedule

func (s tSchedules) proto() []*avia.TSchedule {
	var protoSchedules []*avia.TSchedule
	for _, schedule := range s {
		protoSchedules = append(protoSchedules, schedule.proto())
	}
	return protoSchedules
}

type tFlight struct {
	Title     string
	AirlineID int32
	Schedules tSchedules
}

func (f tFlight) proto() *avia.TFlight {
	return &avia.TFlight{
		Title:     f.Title,
		AirlineID: int64(f.AirlineID),
		Schedules: f.Schedules.proto(),
	}
}

func groupBySchedule(forFlight []segment.RouteSchedule, forFlightErr error) (tSchedules, error) {
	if forFlightErr != nil {
		return nil, forFlightErr
	}

	groupedSchedules := make(map[string][]segment.RouteSchedule)

	for _, schedule := range forFlight {
		hash := schedule.Route.Hash()
		groupedSchedules[hash] = append(groupedSchedules[hash], schedule)
	}

	var schedules []tSchedule
	for _, scheduleGroup := range groupedSchedules {
		if len(scheduleGroup) == 0 {
			continue
		}
		schedule := tSchedule{
			Route: tRoute(scheduleGroup[0].Route),
			Masks: dateMaskToResponseMasks(scheduleGroup),
		}
		schedules = append(schedules, schedule)
	}

	return schedules, nil

}

func dateMaskToResponseMasks(schedules []segment.RouteSchedule) []format.Mask {

	if len(schedules) == 0 {
		return nil
	}
	baseMask := segment.SumUpMasks(schedules)
	return flightp2p.GenerateMasks(baseMask, dtutil.IntDate(-1))
}

func createSchedulesForFlight(
	patterns []*structs.FlightPattern,
	flightStorage iFlightStorage,
) ([]segment.RouteSchedule, error) {
	var schedules []segment.RouteSchedule

	var segments segment.SegmentGroup
	for i := range patterns {
		var flightBase structs.FlightBase
		var err error
		if flightBase, err = flightStorage.GetFlightBase(patterns[i].FlightBaseID, patterns[i].IsDop); err != nil {
			logger.Logger().Warn("Unknown flight base", log.Reflect("pattern", patterns[i]))
		}

		segment := segment.Segment{FlightBase: flightBase, FlightPattern: *(patterns[i])}
		segments = append(segments, segment)
	}

	for schedule := range segment.GenerateSchedules(segments) {
		schedules = append(schedules, schedule)
	}

	return schedules, nil
}

func filterWithoutFlightBase(patterns []*structs.FlightPattern, flightStorage iFlightStorage) error {
	for _, pattern := range patterns {
		if _, err := flightStorage.GetFlightBase(pattern.FlightBaseID, pattern.IsDop); err != nil {
			return xerrors.Errorf("filterWithoutFlightBase: %v %v", pattern.ID, pattern.FlightBaseID)
		}
	}
	return nil
}

func filterDisjointRoutePoints(patterns []*structs.FlightPattern, flightStorage iFlightStorage) error {
	possibleArrivals := make(map[int32]map[int64]struct{}) // map from leg sequence number to station ids
	for _, pattern := range patterns {
		m, ok := possibleArrivals[pattern.LegNumber]
		if !ok {
			m = make(map[int64]struct{})
		}
		flightBase, _ := flightStorage.GetFlightBase(pattern.FlightBaseID, pattern.IsDop)
		m[flightBase.ArrivalStation] = struct{}{}
		possibleArrivals[pattern.LegNumber] = m
	}

	for _, pattern := range patterns {
		m, ok := possibleArrivals[pattern.LegNumber-1]
		if !ok {
			continue
		}
		flightBase, _ := flightStorage.GetFlightBase(pattern.FlightBaseID, pattern.IsDop)
		if _, stationExists := m[flightBase.DepartureStation]; !stationExists {
			return xerrors.Errorf(
				"non matching stations, %v not found in %#v",
				flightBase.DepartureStation,
				m,
			)
		}
	}
	return nil
}

func filterDisjointLegSequence(patterns []*structs.FlightPattern) error {
	var legNumbers uint64
	for _, pattern := range patterns {
		legNumbers |= uint64(1) << pattern.LegNumber
	}
	if (legNumbers & uint64(1)) > 0 {
		return xerrors.Errorf("leg sequence number = 0: %#v", patterns)
	}

	lastOk := true
	for i := 1; i < 64; i++ {
		if (legNumbers & (1 << i)) > 0 {
			if !lastOk {
				return xerrors.Errorf("disjoint leg sequences: %#v", patterns)
			}
			lastOk = true
		} else {
			lastOk = false
		}
	}

	return nil
}
