package flightdata

import (
	"time"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/travel/avia/shared_flights/api/internal/appconst"
	dto "a.yandex-team.ru/travel/avia/shared_flights/api/internal/services/storage/DTO"
	"a.yandex-team.ru/travel/avia/shared_flights/api/pkg/structs"
	"a.yandex-team.ru/travel/avia/shared_flights/lib/go/dtutil"
	"a.yandex-team.ru/travel/avia/shared_flights/lib/go/logger"
)

type TimezoneProvider interface {
	GetTimeZoneByStationID(stationID int64) *time.Location
}
type FlightDataBase struct {
	FlightBase    structs.FlightBase
	FlightPattern *structs.FlightPattern
	FlightStatus  *structs.FlightStatus
	Codeshares    []dto.TitledFlight
	// providers
	tzProvider TimezoneProvider

	departureTimezone    *time.Location
	arrivalTimezone      *time.Location
	arrivalDayDifference *int
}

type FlightData struct {
	*FlightDataBase
	FlightDepartureDate dtutil.IntDate // the date when the first leg of the flight is scheduled to take off

	// service fields
	departureTimeScheduled       time.Time
	actualDepartureTime          time.Time
	actualDepartureTimeScheduled time.Time // scheduled time according to status info
	arrivalTimeScheduled         time.Time
	actualArrivalTime            time.Time
	actualArrivalTimeScheduled   time.Time // scheduled time according to status info
	Banned                       bool      // whether this flight is banned
}

func NewFlightDataBase(
	flightBase structs.FlightBase,
	flightPattern *structs.FlightPattern,
	flightStatus *structs.FlightStatus,
	tzProvider TimezoneProvider,
) *FlightDataBase {
	return &FlightDataBase{
		FlightBase:    flightBase,
		FlightPattern: flightPattern,
		FlightStatus:  flightStatus,
		tzProvider:    tzProvider,
	}
}

func NewFlightData(
	flightBase structs.FlightBase,
	flightPattern *structs.FlightPattern,
	flightStatus *structs.FlightStatus,
	flightDepartureDate dtutil.IntDate,
	tzProvider TimezoneProvider,
) *FlightData {
	return &FlightData{
		FlightDataBase: &FlightDataBase{
			FlightBase:    flightBase,
			FlightPattern: flightPattern,
			FlightStatus:  flightStatus,
			tzProvider:    tzProvider,
		},
		FlightDepartureDate: flightDepartureDate,
	}
}

// Only used to simplify date filter tests
func NewFlightDataForTests(departureTimeScheduled, actualDepartureTime, arrivalTimeScheduled, actualArrivalTime time.Time) *FlightData {
	return &FlightData{
		departureTimeScheduled: departureTimeScheduled,
		actualDepartureTime:    actualDepartureTime,
		arrivalTimeScheduled:   arrivalTimeScheduled,
		actualArrivalTime:      actualArrivalTime,
	}
}

func (fd *FlightDataBase) DepartureTimezone() *time.Location {
	if fd.departureTimezone == nil && fd.tzProvider == nil {
		return nil
	}
	fd.departureTimezone = fd.tzProvider.GetTimeZoneByStationID(fd.FlightBase.DepartureStation)
	return fd.departureTimezone
}

func (fd *FlightDataBase) ArrivalTimezone() *time.Location {
	if fd.arrivalTimezone == nil && fd.tzProvider == nil {
		return nil
	}
	fd.arrivalTimezone = fd.tzProvider.GetTimeZoneByStationID(fd.FlightBase.ArrivalStation)
	return fd.arrivalTimezone
}

func (fd *FlightDataBase) DepartureDayDifference() int {
	if fd.FlightPattern != nil {
		return int(fd.FlightPattern.DepartureDayShift)
	}
	return 0
}

func (fd *FlightDataBase) ArrivalDayDifference(departureDate ...dtutil.IntDate) int {
	if fd.FlightPattern != nil {
		return int(fd.FlightPattern.ArrivalDayShift)
	}

	if fd.arrivalDayDifference != nil {
		return *fd.arrivalDayDifference
	}

	var dd dtutil.IntDate
	if len(departureDate) == 0 {
		if fd.FlightPattern == nil {
			logger.Logger().Error("Cannot determine flight day difference:"+
				"no departure date passed and flight pattern is nil", log.Reflect("fd", fd))
			return 0
		}
		dd = dtutil.StringDate(fd.FlightPattern.OperatingFromDate).ToIntDate()
	} else {
		dd = departureDate[0]
	}
	flightData := FlightData{
		FlightDataBase:      fd,
		FlightDepartureDate: dd,
	}
	departure := flightData.ScheduledDeparture()
	departureInArrivalTz := departure.In(fd.ArrivalTimezone())
	departureYear, departureMonth, departureDay := departureInArrivalTz.Date()
	arrivalDateTimeLocal := time.Time{}
	if dtutil.IntTime(fd.FlightBase.ArrivalTimeScheduled).IsValid() {
		arrivalDateTimeLocal = time.Date(
			departureYear, departureMonth, departureDay,
			int(fd.FlightBase.ArrivalTimeScheduled)/100, int(fd.FlightBase.ArrivalTimeScheduled)%100, 0, 0,
			fd.ArrivalTimezone(),
		)
	}

	for arrivalDateTimeLocal.Before(departureInArrivalTz) {
		arrivalDateTimeLocal = arrivalDateTimeLocal.Add(time.Hour * time.Duration(24))
	}
	diff := int(int64(
		dtutil.TruncateToDateUTC(arrivalDateTimeLocal).
			Sub(dtutil.TruncateToDateUTC(departure)).
			Hours(),
	) / 24)
	fd.arrivalDayDifference = &diff
	return *fd.arrivalDayDifference
}

func (fd *FlightData) ScheduledDeparture() time.Time {
	if !fd.departureTimeScheduled.IsZero() {
		return fd.departureTimeScheduled
	}
	stationTz := fd.DepartureTimezone()
	if stationTz == nil {
		return time.Time{}
	}
	fd.departureTimeScheduled = dtutil.IntToTime(
		fd.FlightDepartureDate, dtutil.IntTime(fd.FlightBase.DepartureTimeScheduled), stationTz,
	)
	return fd.departureTimeScheduled
}

func (fd *FlightData) ScheduledArrival() time.Time {
	if !fd.arrivalTimeScheduled.IsZero() {
		return fd.arrivalTimeScheduled
	}

	stationTz := fd.ArrivalTimezone()
	if stationTz == nil {
		return time.Time{}
	}

	fd.arrivalTimeScheduled = dtutil.IntToTime(
		fd.FlightDepartureDate, dtutil.IntTime(fd.FlightBase.ArrivalTimeScheduled), stationTz,
	).AddDate(0, 0, fd.ArrivalDayDifference(fd.FlightDepartureDate))
	return fd.arrivalTimeScheduled
}

// STATUS

func (fd *FlightData) ActualDeparture() time.Time {
	actual := fd.actualDeparture()
	if actual.IsZero() && fd.FlightStatus != nil && fd.FlightStatus.DepartureSourceID == appconst.AirportFlightStatusSource {
		actual = fd.actualScheduledDeparture()
	}
	return actual
}

func (fd *FlightData) actualDeparture() time.Time {
	if !fd.actualDepartureTime.IsZero() {
		return fd.actualDepartureTime
	}
	stationTz := fd.DepartureTimezone()
	if stationTz == nil {
		return time.Time{}
	}

	if fd.FlightStatus != nil && fd.FlightStatus.DepartureTimeActual != "" {
		t, err := dtutil.ParseDateTimeISO(fd.FlightStatus.DepartureTimeActual, stationTz)
		if err != nil {
			logParseError(fd.FlightStatus.DepartureTimeActual, fd)
		}
		fd.actualDepartureTime = t
	}
	return fd.actualDepartureTime
}

// scheduled departure time according to status info
func (fd *FlightData) actualScheduledDeparture() time.Time {
	if !fd.actualDepartureTimeScheduled.IsZero() {
		return fd.actualDepartureTimeScheduled
	}
	stationTz := fd.DepartureTimezone()
	if stationTz == nil {
		return time.Time{}
	}
	if fd.FlightStatus != nil && fd.FlightStatus.DepartureTimeScheduled != "" {
		t, err := dtutil.ParseDateTimeISO(fd.FlightStatus.DepartureTimeScheduled, stationTz)
		if err != nil {
			logParseError(fd.FlightStatus.DepartureTimeScheduled, fd)
		}
		fd.actualDepartureTimeScheduled = t
	}
	return fd.actualDepartureTimeScheduled
}

func (fd *FlightData) ActualArrival() time.Time {
	actual := fd.actualArrival()
	if actual.IsZero() && fd.FlightStatus != nil && fd.FlightStatus.ArrivalSourceID == appconst.AirportFlightStatusSource {
		actual = fd.actualScheduledArrival()
	}
	return actual
}

func (fd *FlightData) actualArrival() time.Time {
	if !fd.actualArrivalTime.IsZero() {
		return fd.actualArrivalTime
	}

	stationTz := fd.ArrivalTimezone()
	if stationTz == nil {
		return time.Time{}
	}

	if fd.FlightStatus != nil && fd.FlightStatus.ArrivalTimeActual != "" {
		t, err := dtutil.ParseDateTimeISO(fd.FlightStatus.ArrivalTimeActual, stationTz)
		if err != nil {
			logParseError(fd.FlightStatus.ArrivalTimeActual, fd)
		}
		fd.actualArrivalTime = t
	}
	return fd.actualArrivalTime
}

// scheduled arrival time according to status info
func (fd *FlightData) actualScheduledArrival() time.Time {
	if !fd.actualArrivalTimeScheduled.IsZero() {
		return fd.actualArrivalTimeScheduled
	}
	stationTz := fd.ArrivalTimezone()
	if stationTz == nil {
		return time.Time{}
	}
	if fd.FlightStatus != nil && fd.FlightStatus.ArrivalTimeScheduled != "" {
		t, err := dtutil.ParseDateTimeISO(fd.FlightStatus.ArrivalTimeScheduled, stationTz)
		if err != nil {
			logParseError(fd.FlightStatus.ArrivalTimeScheduled, fd)
		}
		fd.actualArrivalTimeScheduled = t
	}
	return fd.actualArrivalTimeScheduled
}

func (fd *FlightData) DepartureStatusIsFromTrustedSource() bool {
	status := fd.FlightStatus
	if status == nil {
		return false
	}
	return status.DepartureSourceIsTrusted
}

func (fd *FlightData) ArrivalStatusIsFromTrustedSource() bool {
	status := fd.FlightStatus
	if status == nil {
		return false
	}
	return status.ArrivalSourceIsTrusted
}

func (fd *FlightDataBase) ForDate(departureDate dtutil.IntDate) FlightData {
	return FlightData{
		FlightDataBase: &FlightDataBase{
			fd.FlightBase,
			fd.FlightPattern,
			fd.FlightStatus,
			[]dto.TitledFlight{},
			fd.tzProvider,
			fd.departureTimezone,
			fd.arrivalTimezone,
			fd.arrivalDayDifference,
		},
		FlightDepartureDate: departureDate,
	}
}

func logParseError(dt string, fd *FlightData) {
	logger.Logger().Error(
		"Cannot parse datetime from status",
		log.String("time scheduled", dt),
		log.String("carrier", fd.FlightBase.OperatingCarrierCode),
		log.String("flight", fd.FlightBase.OperatingFlightNumber),
		log.Int32("date", int32(fd.FlightDepartureDate)),
	)
}
