package loadsnapshot

import (
	"a.yandex-team.ru/travel/avia/shared_flights/api/internal/logthrottler"
	"time"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/core/xerrors"
	"a.yandex-team.ru/travel/avia/shared_flights/api/internal/services/storage/timezone"
	"a.yandex-team.ru/travel/avia/shared_flights/api/internal/storage/flight"
	"a.yandex-team.ru/travel/avia/shared_flights/api/internal/utils"
	"a.yandex-team.ru/travel/avia/shared_flights/api/pkg/structs"
	dir "a.yandex-team.ru/travel/avia/shared_flights/lib/go/direction"
	"a.yandex-team.ru/travel/avia/shared_flights/lib/go/dtutil"
	"a.yandex-team.ru/travel/avia/shared_flights/lib/go/logger"
	"a.yandex-team.ru/travel/proto/shared_flights/snapshots"
)

type flightStatusLoaderArtifacts struct {
	flightStatusesToProcess map[string]FlightStatusInfo
	dopFlightsToCreate      map[string]flight.FlightPatternAndBase
	dopFlightsCounts        utils.StringListMap
	errorCounters           map[string]int
}
type flightStatusLoaderStats struct {
	flightStatusesSkipped     int
	flightStatusesLoadedCount int
}
type flightStatusLoaderConfig struct {
	startScheduleDate          string
	stations                   stationByID
	tzutil                     timezone.TimeZoneProvider
	carrierService             carrierService
	flightStorage              legFinder
	carrierStorage             carrierByCode
	stationStatusSourceStorage trustVerifier
	statusStorage              flightStatusSaver
}

type flightStatusLoader struct {
	flightStatusLoaderConfig
	flightStatusLoaderStats
	flightStatusLoaderArtifacts
	nextDopFlightID int32
}
type carrierService interface {
	GetCarrierByCodeAndFlightNumber(carrierCode, flightNumber string) int32
}
type legFinder interface {
	FindLegInfo(carrierID int32, flightNumber, flightDate string, departureStation, arrivalStation int64, useArrivalShift bool) (legInfo flight.FlightPatternAndBase, err error)
}
type carrierByCode interface {
	GetCarriersByCode(code string) []int32
}
type trustVerifier interface {
	IsTrusted(stationID int64, statusSourceID int64) bool
}
type flightStatusSaver interface {
	PutFlightStatus(flightStatus structs.FlightStatus) structs.FlightStatus
}

func (fl *flightStatusLoader) LoadStatus(
	flightStatusProto *snapshots.TFlightStatus,
) {

	var flightStatus = structs.FlightStatus{}
	structs.FlightStatusFromProto(flightStatusProto, &flightStatus)

	if flightStatus.FlightDate < fl.startScheduleDate {
		fl.flightStatusesSkipped++
		return
	}

	// should return either corrected carrier or first carrier with desired code or 0
	// assuming that carrier codes start with 1
	flightStatus.AirlineID = int64(fl.carrierService.GetCarrierByCodeAndFlightNumber(
		flightStatus.CarrierCode,
		flightStatus.FlightNumber,
	))
	type unknownCarrierCode string
	if flightStatus.AirlineID == 0 {
		fl.errorCounters["error_unknown_code"] += 1
		logthrottler.LogWithThrottling(
			unknownCarrierCode(flightStatus.CarrierCode),
			1*time.Hour,
			logger.Logger().AddCallerSkip(1).Info,
			"Unknown carrier code",
			log.String("carrier", flightStatus.CarrierCode),
			log.Reflect("status", flightStatus),
		)
		return
	}

	// Пробуем найти статус в расписании
	// Вполне возможно, что станции в статусе невалидны, поэтому мы можем доверять
	// только станциям вылета для статуса вылета и станциям прилета для статуса прилета.
	// Вариации следующие:
	// 1. Не можем найти расписание ни для вылета, ни для прилёта. Это чистый доп.
	//     делим статус пополам и загружаем каждую половинку как доп
	// 2. Мы нашли сегмент только для одной половинки:
	//     делим статус пополам и пытаемся загрузить половинку с известным сегментом,
	//     потому что половинка без сегмента может создать конфликтующийи доп-рейс.
	// 3. Нашли для обеих половинок:
	//     делим статус пополам и загружаем каждую половинку со своей информацией о сегменте
	// В любом случае игнорируем статусы, если не нашли точного совпадения по сегментам,
	// но нашли совпадение по номеру рейса на текущую дату

	departureLegInfo, arrivalLegInfo, departureErr, arrivalErr := fl.getLegsInfo(flightStatus)
	var ignoreDeparture = xerrors.Is(departureErr, ignoreStatusError)
	var ignoreArrival = xerrors.Is(arrivalErr, ignoreStatusError)
	if ignoreDeparture && ignoreArrival {
		logger.Logger().Debug(
			"Status should be ignored",
			log.Reflect("code", flightStatus.CarrierCode),
			log.Reflect("number", flightStatus.FlightNumber),
			log.Reflect("date", flightStatus.FlightDate),
			log.NamedError("departure", departureErr),
			log.NamedError("arrival", arrivalErr),
		)
		return
	}

	departureStatus := withDepartureScheduledFlightDate(flightStatus.DepartureCopy())
	arrivalStatus := withArrivalScheduledFlightDate(flightStatus.ArrivalCopy())
	if arrivalLegInfo != nil && arrivalLegInfo.FlightPattern.ArrivalDayShift != 0 {
		newFlightDateIndex, idxOK := dtutil.DateCache.IndexOfStringDate(dtutil.StringDate(arrivalStatus.FlightDate))
		if !idxOK {
			logger.Logger().Error("Cannot update arrival status flight date")
			return
		}
		newFlightDateIndex -= int(arrivalLegInfo.FlightPattern.ArrivalDayShift)
		arrivalStatus.FlightDate = string(dtutil.DateCache.Date(newFlightDateIndex).StringDateDashed())
	}
	if departureLegInfo != nil && arrivalLegInfo != nil || departureLegInfo == nil && arrivalLegInfo == nil {
		if flightStatus.ArrivalStatusExists() && !ignoreArrival {
			fl.tryLoadStatus(arrivalStatus, arrivalLegInfo)
		}
		if flightStatus.DepartureStatusExists() && !ignoreDeparture {
			fl.tryLoadStatus(departureStatus, departureLegInfo)
		}
	} else if departureLegInfo == nil {
		if flightStatus.ArrivalStatusExists() && !ignoreArrival {
			fl.tryLoadStatus(arrivalStatus, arrivalLegInfo)
		}
	} else if arrivalLegInfo == nil {
		if flightStatus.DepartureStatusExists() && !ignoreDeparture {
			fl.tryLoadStatus(departureStatus, departureLegInfo)
		}
	}
}

func (fl *flightStatusLoader) getLegsInfo(flightStatus structs.FlightStatus) (departureLegInfo, arrivalLegInfo *flight.FlightPatternAndBase, departureErr, arrivalErr error) {
	if flightStatus.DepartureStatusExists() {
		if flightStatus.DepartureStation == 0 {
			logger.Logger().Error(
				"Existing departure status without departure station",
				log.Reflect("status", flightStatus),
			)
			departureErr = ignoreStatusError
		} else {
			departureStatusLegInfo, err := fl.getLegInfo(withDepartureScheduledFlightDate(flightStatus), dir.DEPARTURE)
			if err == nil {
				departureLegInfo = &departureStatusLegInfo
			} else {
				fl.errorCounters["error_"+err.Error()] += 1
				if xerrors.Is(err, ignoreStatusError) {
					departureErr = err
				}
			}
		}
	}
	if flightStatus.ArrivalStatusExists() {
		if flightStatus.ArrivalStation == 0 {
			logger.Logger().Error(
				"Existing arrival status without departure station",
				log.Reflect("status", flightStatus),
			)
			arrivalErr = ignoreStatusError
		} else {
			arrivalStatusLegInfo, err := fl.getLegInfo(withArrivalScheduledFlightDate(flightStatus), dir.ARRIVAL)
			if err == nil {
				arrivalLegInfo = &arrivalStatusLegInfo
			} else {
				fl.errorCounters["error_"+err.Error()] += 1
				if xerrors.Is(err, ignoreStatusError) {
					arrivalErr = err
				}
			}
		}
	}
	return departureLegInfo, arrivalLegInfo, departureErr, arrivalErr
}

var ignoreStatusError = xerrors.New("ignore status")

func (fl *flightStatusLoader) getLegInfo(flightStatus structs.FlightStatus, direction dir.Direction) (flight.FlightPatternAndBase, error) {
	var departureStation int64
	var arrivalStation int64
	var flightDate = flightStatus.FlightDate
	var useArrivalShift = false

	switch direction {
	case dir.DEPARTURE:
		departureStation = flightStatus.DepartureStation
	case dir.ARRIVAL:
		arrivalStation = flightStatus.ArrivalStation
		useArrivalShift = true
	default:
		logger.Logger().Error("Unknown direction", log.String("direction", direction.String()))
	}
	legInfo, err := fl.flightStorage.FindLegInfo(
		int32(flightStatus.AirlineID),
		flightStatus.FlightNumber,
		flightDate,
		departureStation,
		arrivalStation,
		useArrivalShift,
	)
	if err == nil {
		return legInfo, nil
	}
	carrierCandidates := fl.carrierStorage.GetCarriersByCode(flightStatus.CarrierCode)
	if len(carrierCandidates) == 0 {
		err = xerrors.New("unknown carrier code")
		fl.errorCounters[err.Error()] += 1
		logger.Logger().Warn("Unknown carrier code", log.String("carrier", flightStatus.CarrierCode))
		return flight.FlightPatternAndBase{}, err
	}
	for _, carrierID := range carrierCandidates {
		// try carrier code without correction
		legInfo, err = fl.flightStorage.FindLegInfo(
			carrierID,
			flightStatus.FlightNumber,
			flightDate,
			departureStation,
			arrivalStation,
			useArrivalShift,
		)
		if err != nil {
			fl.errorCounters[err.Error()] += 1
			continue
		}
		flightStatus.AirlineID = int64(carrierID)
		logger.Logger().Warn(
			"Found flight info while scanning through possible carriers",
			// This should not have happened, because all patterns should be iata-corrected and
			// carrier codes should not be ambiguous.
			log.Reflect("status airline id", flightStatus.AirlineID),
			log.Reflect("status airline code", flightStatus.CarrierCode),
			log.Reflect("status flight number", flightStatus.FlightNumber),
			log.Int32("carrier", carrierID),
		)
		return legInfo, nil
	}

	if _, err := fl.flightStorage.FindLegInfo(int32(flightStatus.AirlineID), flightStatus.FlightNumber, flightDate, 0, 0, useArrivalShift); err == nil {
		return flight.FlightPatternAndBase{}, xerrors.Errorf(
			"dop flight should be ignored: non-dop segments on that day: %w",
			ignoreStatusError,
		)
	}

	return flight.FlightPatternAndBase{}, xerrors.New("leg info not found")
}

func (fl *flightStatusLoader) tryLoadStatus(flightStatus structs.FlightStatus, legInfo *flight.FlightPatternAndBase) {
	if legInfo != nil {
		flightStatus.CarrierCode = legInfo.FlightPattern.MarketingCarrierCode
		flightStatus.AirlineID = int64(legInfo.FlightPattern.MarketingCarrier)
		flightStatus.DepartureStation = legInfo.FlightBase.DepartureStation
		flightStatus.ArrivalStation = legInfo.FlightBase.ArrivalStation
		flightStatus.LegNumber = legInfo.FlightPattern.LegNumber

		if legInfo.FlightPattern.LegNumber == 0 || legInfo.FlightBase.LegNumber == 0 {
			logger.Logger().Error(
				"Leg number should not be zero, but it is",
				log.Reflect("status", flightStatus),
				log.Reflect("leg info", legInfo),
			)
			return
		}
		if legInfo.FlightPattern.LegNumber != legInfo.FlightBase.LegNumber {
			logger.Logger().Error(
				"Leg number doesnt match",
				log.Reflect("status", flightStatus),
				log.Reflect("leg info", legInfo),
			)
			return
		}
	} else {
		flightStatus.LegNumber = 1
	}

	fl.flightStatusesLoadedCount++

	flightStatusKey := getFlightStatusKey(flightStatus)

	flightStatus.FillSourceTrust(fl.stationStatusSourceStorage)
	flightStatus = fl.statusStorage.PutFlightStatus(flightStatus)

	// Use the most updated flight status for delays/cancellations counts,
	// just the last-processed one is not enough, it might have only half of data
	if legInfo != nil {
		scheduledDeparture := ""
		scheduledArrival := ""
		departureTz := fl.tzutil.GetTimeZoneByStationID(legInfo.FlightBase.DepartureStation)
		arrivalTz := fl.tzutil.GetTimeZoneByStationID(legInfo.FlightBase.ArrivalStation)
		if departureTz != nil && arrivalTz != nil {
			scheduledDeparture = scheduledTime(
				flightStatus.FlightDate,
				0,
				legInfo.FlightBase.DepartureTimeScheduled,
				departureTz,
			)
			scheduledArrival = scheduledTime(
				flightStatus.FlightDate,
				int(legInfo.FlightPattern.ArrivalDayShift),
				legInfo.FlightBase.ArrivalTimeScheduled,
				arrivalTz,
			)
		}
		fl.flightStatusesToProcess[flightStatusKey] = FlightStatusInfo{
			FlightStatus:       flightStatus,
			FlightBase:         legInfo.FlightBase,
			FlightPattern:      legInfo.FlightPattern,
			ScheduledDeparture: scheduledDeparture,
			ScheduledArrival:   scheduledArrival,
		}
	}

	// No luck in searching for the matching flight ==> create a dop flight
	if legInfo == nil &&
		flightStatus.DepartureStation > 0 &&
		flightStatus.ArrivalStation > 0 &&
		(len(flightStatus.DepartureTimeScheduled) > 0 || len(flightStatus.ArrivalTimeScheduled) > 0) {
		fl.nextDopFlightID++
		flightBase, flightPattern, ok := CreateDopFlight(
			flightStatus,
			fl.nextDopFlightID,
			fl.tzutil,
			fl.stations,
		)
		if ok {
			flightStations := getFlightStations(&flightBase)
			fl.dopFlightsCounts.Put(flightStatusKey, flightStations)
			fl.dopFlightsToCreate[flightStatusKey] = flight.FlightPatternAndBase{
				FlightBase:    flightBase,
				FlightPattern: flightPattern,
			}
			if _, knownLeg := fl.flightStatusesToProcess[flightStatusKey]; !knownLeg {
				fl.flightStatusesToProcess[flightStatusKey] = FlightStatusInfo{
					FlightStatus:       flightStatus,
					FlightBase:         flightBase,
					FlightPattern:      flightPattern,
					ScheduledDeparture: flightStatus.DepartureTimeScheduled,
					ScheduledArrival:   flightStatus.ArrivalTimeScheduled,
				}
			}
		}
	}
}

func withDepartureScheduledFlightDate(fs structs.FlightStatus) structs.FlightStatus {
	t, err := dtutil.ParseDateTimeISO(fs.DepartureTimeScheduled, time.UTC)
	if err == nil {
		fs.FlightDate = string(dtutil.FormatDateIso(t))
	}
	return fs
}

func withArrivalScheduledFlightDate(fs structs.FlightStatus) structs.FlightStatus {
	t, err := dtutil.ParseDateTimeISO(fs.ArrivalTimeScheduled, time.UTC)
	if err == nil {
		fs.FlightDate = string(dtutil.FormatDateIso(t))
	}
	return fs
}
