package loadsnapshot

import (
	"archive/zip"
	"bytes"
	"encoding/binary"
	"fmt"
	"io"
	"math"
	"sort"
	"strings"
	"time"

	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/aws/credentials"
	"github.com/aws/aws-sdk-go/aws/session"
	"github.com/aws/aws-sdk-go/service/s3"
	"github.com/aws/aws-sdk-go/service/s3/s3manager"
	"github.com/golang/protobuf/proto"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/core/xerrors"
	aviaMetrics "a.yandex-team.ru/travel/avia/library/go/metrics"
	"a.yandex-team.ru/travel/avia/shared_flights/api/internal/config"
	"a.yandex-team.ru/travel/avia/shared_flights/api/internal/metrics"
	storageServices "a.yandex-team.ru/travel/avia/shared_flights/api/internal/services/storage"
	"a.yandex-team.ru/travel/avia/shared_flights/api/internal/services/storage/carrier"
	"a.yandex-team.ru/travel/avia/shared_flights/api/internal/services/storage/timezone"
	"a.yandex-team.ru/travel/avia/shared_flights/api/internal/storage"
	"a.yandex-team.ru/travel/avia/shared_flights/api/internal/storage/flight"
	"a.yandex-team.ru/travel/avia/shared_flights/api/internal/storage/status"
	"a.yandex-team.ru/travel/avia/shared_flights/api/internal/utils"
	"a.yandex-team.ru/travel/avia/shared_flights/api/pkg/structs"
	"a.yandex-team.ru/travel/avia/shared_flights/lib/go/dtutil"
	iatacorrector "a.yandex-team.ru/travel/avia/shared_flights/lib/go/iata_correction"
	"a.yandex-team.ru/travel/avia/shared_flights/lib/go/logger"
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/helpers"
	"a.yandex-team.ru/travel/proto/dicts/rasp"
	"a.yandex-team.ru/travel/proto/shared_flights/snapshots"
	"a.yandex-team.ru/travel/proto/shared_flights/ssim"
)

// When we read something from the incoming binary snapshot file and that something tells us
// out protobuf structure size is greater than this value, it's a definite sign something went wrong
const MaxProtoSize = 100000

const flightStatusProtoDatetimeFormat = "2006-01-02 15:04:05"

type Service interface {
	LoadSnapshot(
		currentBase string, // skip loading base snapshot if the latest base snapshot marker is the same as the current one
		currentDelta string, // skip loading delta snapshot if the latest delat snapshot marker is the same as the current one
		skipBase bool, // for invsetigation purposes only: skip loading base snapshot, no matter what the latest marker value is
	) (result *Result, e error)
}

type LoadSnapshotFromMdsService struct {
	Service *storageServices.Service
	config  *config.Config
}

type Result struct {
	BaseMarker  string
	DeltaMarker string
	DebugInfo   map[string]string // for investigation purposes only, shall not be referenced in the source code
}

type DopFlightKey struct {
	MarketingCarrier      int32
	MarketingFlightNumber string
	LegNumber             int32
	DepartureStation      int64
	ArrivalStation        int64
	FlightDate            string
}

type FlightStatusInfo struct {
	FlightStatus       structs.FlightStatus
	FlightBase         structs.FlightBase
	FlightPattern      structs.FlightPattern
	ScheduledDeparture string
	ScheduledArrival   string
}

func NewLoadSnapshotService(storageService *storageServices.Service, config *config.Config) Service {
	return &LoadSnapshotFromMdsService{
		Service: storageService,
		config:  config,
	}
}

// TODO(u-jeen): make the snapshot/delta load transactional, i.e. load the data in full or discard, no partial loads
func (loader *LoadSnapshotFromMdsService) LoadSnapshot(
	currentBase string, currentDelta string, skipBase bool) (result *Result, e error) {

	result = &Result{
		BaseMarker:  "",
		DeltaMarker: "",
		DebugInfo:   map[string]string{},
	}

	defer func() {
		if panicError := recover(); panicError != nil {
			e = xerrors.Errorf(
				"error loading snapshot: %+v: %+v",
				panicError,
				helpers.GetTraceback(),
			)
		}
	}()

	awsConfig := loader.config.AwsConfig
	// Takes values from env vars AWS_ACCESS_KEY, AWS_SECRET_KEY
	s, sessionErr := session.NewSession(&aws.Config{
		Endpoint:    aws.String(awsConfig.Endpoint),
		Region:      aws.String(awsConfig.Region),
		Credentials: credentials.NewEnvCredentials(),
		MaxRetries:  aws.Int(awsConfig.MaxRetries),
	})
	if sessionErr != nil {
		return nil, exitErrorf("Unable to establish s3/mds session", sessionErr)
	}

	downloader := s3manager.NewDownloader(s)
	strBase, strDelta, e := loader.loadLatestMarkers(downloader, result)
	if e != nil {
		return nil, e
	}
	result.BaseMarker = strBase
	result.DeltaMarker = strDelta

	newStorage := loader.Service.Instance().Storage()
	logger.Logger().Debug("Trying to update current storage")

	finalState := false
	// This way we can control the resulting state of the storage by modifying the local finalState variable
	loaderStartTime := time.Now()
	shouldLoadBaseSnapshot := !skipBase && currentBase != strBase
	shouldLoadDeltaSnapshot := currentDelta != strDelta || shouldLoadBaseSnapshot
	defer func() {
		if shouldLoadBaseSnapshot || shouldLoadDeltaSnapshot {
			aviaMetrics.GlobalMetrics().Timer(metrics.LoaderPrefix, metrics.TotalLoadTime).
				RecordDuration(time.Since(loaderStartTime))
		}
		logger.Logger().Info("Setting storage state", log.Bool("state", finalState))
		loader.Service.Instance().Storage().IsAvailable = finalState
		finalStateFloat := 0.
		if finalState {
			finalStateFloat = 1.
		}
		aviaMetrics.GlobalMetrics().Gauge(metrics.LoaderPrefix, metrics.StorageOk).Set(finalStateFloat)
	}()

	// Load the base snapshot if needed
	if shouldLoadBaseSnapshot {
		// Once we started loading something, the state of the cache is not valid until we finish loading delta
		finalState = false
		logger.Logger().Debug("New base snapshot arrived, so new storage would be created")
		aviaMetrics.GlobalMetrics().Gauge(metrics.LoaderPrefix, metrics.LoadingBaseSnapshot).Set(1)
		baseSnapshotStartLoadTime := time.Now()
		newStorage, e = loader.loadBaseSnapshot(downloader, strBase, result)
		aviaMetrics.GlobalMetrics().Timer(metrics.LoaderPrefix, metrics.BaseLoadTime).
			RecordDuration(time.Since(baseSnapshotStartLoadTime))
		aviaMetrics.GlobalMetrics().Gauge(metrics.LoaderPrefix, metrics.LoadingBaseSnapshot).Set(0)
		if e != nil {
			return nil, xerrors.Errorf("cannot load base snapshot: %w", e)
		}
		logger.Logger().Info("Updated storage caches for the base snapshot", log.Time("timestamp", time.Now()))
	}

	// Load delta snapshot if needed
	if shouldLoadDeltaSnapshot {
		finalState = false
		aviaMetrics.GlobalMetrics().Gauge(metrics.LoaderPrefix, metrics.LoadingDeltaSnapshot).Set(1)
		deltaSnapshotStartLoadTime := time.Now()
		newStorage, e = loader.loadDeltaSnapshot(downloader, newStorage, strBase, strDelta, result)
		aviaMetrics.GlobalMetrics().Timer(metrics.LoaderPrefix, metrics.DeltaLoadTime).
			RecordDuration(time.Since(deltaSnapshotStartLoadTime))
		aviaMetrics.GlobalMetrics().Gauge(metrics.LoaderPrefix, metrics.LoadingDeltaSnapshot).Set(0)
		if e != nil {
			return nil, xerrors.Errorf("cannot load delta snapshot: %w", e)
		}
	}

	if shouldLoadBaseSnapshot || shouldLoadDeltaSnapshot {
		finalState = false
		if e = newStorage.UpdateCacheDependentData(); e != nil {
			return nil, xerrors.Errorf("cannot update cache dependent data: %w", e)
		}
	}

	logger.Logger().Debug("Switching service to new storage")
	loader.Service.SwitchTo(storageServices.NewStorageService(newStorage))

	finalState = true
	return result, nil
}

// Load the latest marker values (for both base and delta snapshot markers)
func (loader *LoadSnapshotFromMdsService) loadLatestMarkers(
	downloader *s3manager.Downloader, result *Result) (strBase string, strDelta string, err error) {
	// First, load the "latest markers" file itself
	awsConfig := loader.config.AwsConfig
	buf := aws.NewWriteAtBuffer([]byte{})
	numBytes, err := downloader.Download(buf,
		&s3.GetObjectInput{
			Bucket: aws.String(awsConfig.Bucket),
			Key:    aws.String(fmt.Sprintf("%v/%v", awsConfig.BucketFolder, awsConfig.LatestFileName)),
		})
	if err != nil {
		return "", "", exitErrorf("Unable to download base/delta numbers", err)
	}
	result.DebugInfo[awsConfig.LatestFileName] = string(buf.Bytes()[:numBytes])
	strLatest := string(buf.Bytes()[:numBytes])

	// Now, parse the contents of the file
	baseMarker := "base="
	strBase = ""
	posBase := strings.Index(strLatest, baseMarker)
	if posBase != -1 {
		strLatest = strLatest[posBase:]
		posSpace := strings.Index(strLatest, " ")
		if posSpace != -1 {
			strBase = strLatest[len(baseMarker):posSpace]
		}
	}
	if strBase == "" {
		return "", "", xerrors.New("unable to load the latest marker for the base snapshot")
	}

	deltaMarker := "delta="
	posDelta := strings.Index(strLatest, deltaMarker)
	strDelta = ""
	if posDelta != -1 {
		strLatest = strLatest[posDelta:]
		posSpace := strings.Index(strLatest, " ")
		if posSpace != -1 {
			strDelta = strLatest[len(deltaMarker):posSpace]
		} else {
			strDelta = strLatest[len(deltaMarker):]
		}
	}
	if strDelta == "" {
		return "", "", xerrors.New("unable to load the latest marker for the delta snapshot")
	}

	return strBase, strDelta, nil
}

func (loader *LoadSnapshotFromMdsService) loadBaseSnapshot(
	downloader *s3manager.Downloader, strBase string, result *Result) (updatedStorage *storage.Storage, err error) {
	logger.Logger().Debug("creating new storage")
	newStorage := storage.NewStorage()
	newStorage.FlightStorage().SetFlightCacheConfig(loader.config.FlightCacheConfig)
	awsConfig := loader.config.AwsConfig
	zipFileBuffer := aws.NewWriteAtBuffer([]byte{})
	logger.Logger().Debug(
		"Downloading file",
		log.String("path", path(awsConfig.BucketFolder, strBase, awsConfig.BaseSnapshotFileName)),
	)
	numBytes, err := downloader.Download(zipFileBuffer,
		&s3.GetObjectInput{
			Bucket: aws.String(awsConfig.Bucket),
			Key:    aws.String(path(awsConfig.BucketFolder, strBase, awsConfig.BaseSnapshotFileName)),
		})
	if err != nil {
		return nil, exitErrorf(fmt.Sprintf("Unable to download base snapshot %v", strBase), err)
	}
	// Open snapshot zip file
	zreader, err := zip.NewReader(bytes.NewReader(zipFileBuffer.Bytes()), numBytes)
	if err != nil {
		return nil, exitErrorf("Unable to open snapshot as zip file", err)
	}
	iataCorrectionRules := []*snapshots.TIataCorrectionRule{}
	carriersByIata := make(map[string]int32)
	designatedCarriers := make(map[int32]string)
	stationResolver := newStorage.Stations().GetCode
	carrierResolver := newStorage.CarrierStorage().GetCarrierByID

	for _, zipFile := range zreader.File {
		var fhandle io.ReadCloser
		logger.Logger().Info("Reading snapshot file", log.String("filename", zipFile.Name))
		counter := 0
		fhandle, err = zipFile.Open()
		if err != nil {
			return nil, exitErrorf(fmt.Sprintf("Unable to open zip entry %v", zipFile.Name), err)
		}
		logger.Logger().Debug("Opened zip file", log.Reflect("file", zipFile))

		writeProgress := progressWriter(
			"Reading file", zipFile.Name, 100*1000*1000 /* each 100 Mb*/, logger.Logger().Debug)
		for {
			var pbSize int32
			err = binary.Read(fhandle, binary.LittleEndian, &pbSize)
			if err == io.EOF {
				logger.Logger().Debug("End of file")
				break
			}
			if err != nil {
				return nil, exitErrorf(fmt.Sprintf("Error while reading line %v", counter), err)
			}
			if pbSize <= 0 || pbSize > MaxProtoSize {
				return nil, exitErrorf(fmt.Sprintf("Invalid protobuf size: %v", pbSize), nil)
			}
			buf := make([]byte, pbSize)
			sizeRead, lineErr := io.ReadAtLeast(fhandle, buf, int(pbSize))
			if lineErr != nil {
				return nil, exitErrorf(fmt.Sprintf("Error while reading bytes in line %v", counter), lineErr)
			}

			writeProgress(sizeRead)

			if zipFile.Name == "designated_carriers.pb2.bin" {
				// parse designated carriers
				var designatedCarrier ssim.TDesignatedCarrier
				parseErr := proto.Unmarshal(buf, &designatedCarrier)
				if parseErr != nil {
					return nil, exitErrorf(fmt.Sprintf("Error while parsing the designated carrier proto  %v", counter), parseErr)
				}
				designatedCarriers[designatedCarrier.Id] = designatedCarrier.Title
				counter++
			} else if zipFile.Name == "carriers.pb2.bin" {
				// parse carriers
				var carrier rasp.TCarrier
				parseErr := proto.Unmarshal(buf, &carrier)
				if parseErr != nil {
					return nil, exitErrorf(fmt.Sprintf("Error while parsing the carrier proto %v", counter), parseErr)
				}
				carriersByIata[carrier.Iata] = carrier.Id
				newStorage.CarrierStorage().PutCarrier(carrier.Id, carrier.Iata, carrier.SirenaId, carrier.Icao, carrier.IcaoRu)
				counter++
			} else if zipFile.Name == "popular_scores.pb2.bin" {
				// parse poluarity scores for the carriers
				var carrierPopularityScore ssim.TCarrierPopularScore
				parseErr := proto.Unmarshal(buf, &carrierPopularityScore)
				if parseErr != nil {
					return nil, exitErrorf(fmt.Sprintf("Error while parsing the carrier popularity score %v", counter), parseErr)
				}
				for _, scoreData := range carrierPopularityScore.PopularScores {
					newStorage.CarriersPopularityScores().SetScore(
						carrierPopularityScore.CarrierId,
						scoreData.NationalVersion,
						scoreData.Score,
					)
					newStorage.CarriersPopularityScores().UpdateDefaultScore(carrierPopularityScore.CarrierId)
				}
				counter++
			} else if zipFile.Name == "iata_correction_rules.pb2.bin" {
				// parse iata correction rule
				var rule snapshots.TIataCorrectionRule
				parseErr := proto.Unmarshal(buf, &rule)
				if parseErr != nil {
					return nil, exitErrorf(fmt.Sprintf("Error while parsing the iata correction rule proto %v", counter), parseErr)
				}
				iataCorrectionRules = append(iataCorrectionRules, &rule)
				counter++
			} else if zipFile.Name == "flight_bases.pb2.bin" {
				iataCorrector := iatacorrector.NewIataCorrector(
					iataCorrectionRules,
					designatedCarriers,
					carriersByIata,
				)
				newStorage.SetIataCorrector(iataCorrector)

				var flightBase ssim.TFlightBase
				parseErr := proto.Unmarshal(buf, &flightBase)
				if parseErr != nil {
					return nil, exitErrorf(fmt.Sprintf("Error while parsing the flight_base proto struct %v", counter), parseErr)
				}
				newStorage.PutFlightBase(structs.FlightBaseFromProto(&flightBase, stationResolver, carrierResolver))
				counter++
			} else if zipFile.Name == "flight_patterns.pb2.bin" {
				// parse flight_patterns
				var flightPattern ssim.TFlightPattern
				parseErr := proto.Unmarshal(buf, &flightPattern)
				if parseErr != nil {
					return nil, exitErrorf(fmt.Sprintf("Error while parsing the flight_pattern proto struct %v", counter), parseErr)
				}
				newStorage.PutFlightPattern(structs.FlightPatternFromProto(&flightPattern, carrierResolver))
				counter++
			} else if zipFile.Name == "timezones.pb2.bin" {
				// parse timezones
				var timezone rasp.TTimeZone
				parseErr := proto.Unmarshal(buf, &timezone)
				if parseErr != nil {
					return nil, exitErrorf(fmt.Sprintf("Error while parsing the timezone proto struct %v", counter), parseErr)
				}
				newStorage.Timezones().PutTimezone(&timezone)
				counter++
			} else if zipFile.Name == "stations_with_codes.pb2.bin" {
				// parse station
				var station snapshots.TStationWithCodes
				parseErr := proto.Unmarshal(buf, &station)
				if parseErr != nil {
					return nil, exitErrorf(fmt.Sprintf("Error while parsing the station proto struct %v", counter), parseErr)
				}
				newStorage.PutStation(&station)
				counter++
			} else if zipFile.Name == "p2p_cache.pb2.bin" {
				// parse p2p-cache
				var entry snapshots.TP2PCacheEntry
				parseErr := proto.Unmarshal(buf, &entry)
				if parseErr != nil {
					return nil, exitErrorf(fmt.Sprintf("Error while parsing the p2p cache proto %v", counter), parseErr)
				}
				newStorage.UpdateP2PCache(&entry)
				counter++
			} else if zipFile.Name == "transport_models.pb2.bin" {
				// parse transport models
				var transportModel rasp.TTransportModel
				parseErr := proto.Unmarshal(buf, &transportModel)
				if parseErr != nil {
					return nil, exitErrorf(fmt.Sprintf("Error while parsing transport model proto %v", counter), parseErr)
				}
				newStorage.UpdateTransportModelsCache(&transportModel)
				counter++
			} else if zipFile.Name == "last_imported.pb2.bin" {
				// parse last imported info
				var lastImported snapshots.TLastImportedInfo
				parseErr := proto.Unmarshal(buf, &lastImported)
				if parseErr != nil {
					return nil, exitErrorf(fmt.Sprintf("Error while parsing the last imported proto %v", counter), parseErr)
				}
				newStorage.UpdateLastImported(lastImported.ImportedDate)
				counter++
			} else {
				if counter == 0 {
					logger.Logger().Info("Skipping unexpected file", log.String("filename", zipFile.Name))
				}
				counter++
			}
		}

		if zipFile.Name != "" {
			result.DebugInfo[zipFile.Name] = fmt.Sprintf("%v", counter)
			writeMetrics(zipFile.Name, counter, zipFile.UncompressedSize64)
		}
		logger.Logger().Info("Parsed lines", log.Int("count", counter))
	}
	if newStorage.GetLastImported() == "" {
		// We don't know when we've last synced the data
		newStorage.UpdateLastImported(string(dtutil.FormatDateTimeISO(time.Now().Add(-1 * 62 * 24 * time.Hour))))
	}
	logger.Logger().Info("Last imported", log.Reflect("lastImported", newStorage.GetLastImported()))

	err = newStorage.BuildAeroflotCache()
	if err != nil {
		return nil, exitErrorf("Error while building aeroflot cache", err)
	}
	err = newStorage.CarriersPopularityScores().UpdateFlightNumbersCache(newStorage.FlightStorage())
	if err != nil {
		return nil, exitErrorf("Error while building flight numbers popularity cache", err)
	}
	return newStorage, nil
}

func (loader *LoadSnapshotFromMdsService) loadDeltaSnapshot(
	downloader *s3manager.Downloader, storage *storage.Storage, strBase string, strDelta string, result *Result) (*storage.Storage, error) {
	storage = storage.Copy()

	awsConfig := loader.config.AwsConfig
	zipFileBuffer := aws.NewWriteAtBuffer([]byte{})
	deltaFileName := fmt.Sprintf(awsConfig.DeltaSnapshotFilePattern, strDelta)
	logger.Logger().Debug(
		"Downloading file",
		log.String("path", path(awsConfig.BucketFolder, strBase, deltaFileName)),
	)
	numBytes, err := downloader.Download(zipFileBuffer,
		&s3.GetObjectInput{
			Bucket: aws.String(awsConfig.Bucket),
			Key:    aws.String(path(awsConfig.BucketFolder, strBase, deltaFileName)),
		})
	if err != nil {
		return nil, exitErrorf(fmt.Sprintf("Unable to download delta snapshot %v/delta-%v", strBase, strDelta), err)
	}
	// Open delta zip file
	zreader, err := zip.NewReader(bytes.NewReader(zipFileBuffer.Bytes()), numBytes)
	if err != nil {
		return nil, exitErrorf("Unable to open delta as zip file", err)
	}
	blacklistRuleStorage := flight.NewBlacklistRuleStorage(storage.Stations())
	newDopFlightsStorage := flight.NewDopFlightStorage(storage.IataCorrector(), storage.CarrierStorage(), "")
	flightMergeRuleStorage := flight.NewFlightMergeRuleStorage()
	tzutil := timezone.NewTimeZoneProvider(storage.Timezones(), storage.Stations())
	statusStorage := status.NewStatusStorage(tzutil)
	stationStatusSourceStorage := status.NewStationStatusSourceStorage()

	skippedFiles := make(map[string]bool)

	// ignore schedules older than 31 days in the past
	startScheduleDate := time.Now().Add(-24 * 31 * time.Hour).Format(dtutil.IsoDate)
	carrierFinder := carrier.NewCarrierService(storage.CarrierStorage(), storage.IataCorrector())

	statusLoader := flightStatusLoader{
		flightStatusLoaderConfig: flightStatusLoaderConfig{
			startScheduleDate:          startScheduleDate,
			stations:                   storage.Stations(),
			tzutil:                     tzutil,
			carrierService:             carrierFinder,
			flightStorage:              storage.FlightStorage(),
			carrierStorage:             storage.CarrierStorage(),
			stationStatusSourceStorage: stationStatusSourceStorage,
			statusStorage:              statusStorage,
		},
		flightStatusLoaderArtifacts: flightStatusLoaderArtifacts{
			flightStatusesToProcess: make(map[string]FlightStatusInfo),
			dopFlightsToCreate:      make(map[string]flight.FlightPatternAndBase),
			dopFlightsCounts:        make(utils.StringListMap),
			errorCounters:           make(map[string]int),
		},
		nextDopFlightID: 1,
	}

	// TODO (mikhailche): изменить подход к загрузке - зависимости между хранилищами долджны явным образом влиять на порядок загрузки файлов, а не наоборот
	fileOrder := map[string]int{
		"status_sources.pb2.bin":        10,
		"station_status_source.pb2.bin": 15,
		"flight_merge_rules.pb2.bin":    20,
		"blacklist_rules.pb2.bin":       30,
		"flight_statuses.pb2.bin":       40,
		"overrides.pb2.bin":             50,
	}
	zipFiles := make([]*zip.File, 0)
	zipFiles = append(zipFiles, zreader.File...)
	sort.Slice(zipFiles, func(i, j int) bool {
		order1, ok1 := fileOrder[zipFiles[i].Name]
		if !ok1 {
			order1 = math.MaxInt16
		}
		order2, ok2 := fileOrder[zipFiles[j].Name]
		if !ok2 {
			order2 = math.MaxInt16
		}
		return order1 < order2
	})

	// TODO (mikhailche): поддержать общий интерфейс загрузки прото файлов https://a.yandex-team.ru/arc/trunk/arcadia/travel/library/go/dicts/base/base.go
	for _, zipFile := range zipFiles {
		var fhandle io.ReadCloser
		logger.Logger().Info("Reading delta file", log.String("filename", zipFile.Name))
		counter := 0
		fhandle, err = zipFile.Open()
		if err != nil {
			return nil, exitErrorf(fmt.Sprintf("Unable to open zip entry %v", zipFile.Name), err)
		}
		for {
			var pbSize int32
			err = binary.Read(fhandle, binary.LittleEndian, &pbSize)
			if err == io.EOF {
				break
			}
			if err != nil {
				return nil, exitErrorf(fmt.Sprintf("Error while reading line %v", counter), err)
			}
			if pbSize <= 0 || pbSize > MaxProtoSize {
				return nil, exitErrorf(fmt.Sprintf("Invalid protobuf size: %v", pbSize), nil)
			}
			buf := make([]byte, pbSize)
			_, lineErr := io.ReadAtLeast(fhandle, buf, int(pbSize))
			if lineErr != nil {
				return nil, exitErrorf(fmt.Sprintf("Error while reading bytes in line %v", counter), lineErr)
			}

			if zipFile.Name == "status_sources.pb2.bin" {
				// parse flight status source (global)
				var flightStatusSource snapshots.TFlightStatusSource
				parseErr := proto.Unmarshal(buf, &flightStatusSource)
				if parseErr != nil {
					return nil, exitErrorf(fmt.Sprintf("Error while parsing the flight status source struct %v", counter), parseErr)
				}
				statusStorage.PutStatusSource(structs.FlightStatusSourceFromProto(&flightStatusSource))
				counter++
			} else if zipFile.Name == "station_status_source.pb2.bin" {
				var stationStatusSource snapshots.TStationStatusSource
				parseErr := proto.Unmarshal(buf, &stationStatusSource)
				if parseErr != nil {
					return nil, exitErrorf(fmt.Sprintf("Error while parsing the flight status source struct %v", counter), parseErr)
				}
				stationStatusSourceStorage.PutStationStatusSourceMapping(&stationStatusSource)
				counter++
			} else if zipFile.Name == "flight_statuses.pb2.bin" {
				counter++
				var flightStatusProto snapshots.TFlightStatus
				parseErr := proto.Unmarshal(buf, &flightStatusProto)
				if parseErr != nil {
					return nil, exitErrorf(fmt.Sprintf("Error while parsing the flight_status proto struct %v", counter), parseErr)
				}
				statusLoader.LoadStatus(&flightStatusProto)
			} else if zipFile.Name == "overrides.pb2.bin" {
				var override snapshots.TOverride
				parseErr := proto.Unmarshal(buf, &override)
				if parseErr != nil {
					return nil, exitErrorf(fmt.Sprintf("Error while parsing the override proto struct %v", counter), parseErr)
				}
				// TODO(u-jeen): place overrides into the storage, don't ignore them
				counter++
			} else if zipFile.Name == "blacklist_rules.pb2.bin" {
				// parse blacklist rule
				var rule snapshots.TBlacklistRule
				parseErr := proto.Unmarshal(buf, &rule)
				if parseErr != nil {
					return nil, exitErrorf(fmt.Sprintf("Error while parsing the blacklist rule proto %v", counter), parseErr)
				}
				// cut out time from the flight date restrictions
				if len(rule.FlightDateSince) > 10 {
					rule.FlightDateSince = rule.FlightDateSince[:10]
				}
				if len(rule.FlightDateUntil) > 10 {
					rule.FlightDateUntil = rule.FlightDateUntil[:10]
				}
				blacklistRuleStorage.AddRule(&rule)
				counter++
			} else if zipFile.Name == "flight_merge_rules.pb2.bin" {
				// parse flight merge rules
				var flightMergeRule snapshots.TFlightMergeRule
				parseErr := proto.Unmarshal(buf, &flightMergeRule)
				if parseErr != nil {
					return nil, exitErrorf(fmt.Sprintf("Error while parsing the flight merge rule struct %v", counter), parseErr)
				}
				flightMergeRuleStorage.AddRule(structs.FlightMergeRuleFromProto(&flightMergeRule))
				counter++
			} else {
				if _, ok := skippedFiles[zipFile.Name]; !ok {
					logger.Logger().Info("Skipping unexpected file", log.String("filename", zipFile.Name))
					skippedFiles[zipFile.Name] = true
				}
			}
		}

		if zipFile.Name != "" {
			result.DebugInfo[zipFile.Name] = fmt.Sprintf("%v", counter)
			writeMetrics(zipFile.Name, counter, zipFile.UncompressedSize64)
		}
		logger.Logger().Info("Parsed lines", log.Int("count", counter))
	}

	dopFlightsCreatedCount := 0
	dopFlightsMultilegCount := 0

	for key, flt := range statusLoader.dopFlightsToCreate {
		if statusLoader.dopFlightsCounts.GetCount(key) > 1 {
			dopFlightsMultilegCount++
			continue
		}
		dopFlightsCreatedCount++
		newDopFlightsStorage.PutDopFlightBase(flt.FlightBase)
		newDopFlightsStorage.PutDopFlightPattern(flt.FlightPattern)
	}

	for key, flightStatusInfo := range statusLoader.flightStatusesToProcess {
		// skip multilegs, since we are not showing them correctly anyway
		if statusLoader.dopFlightsCounts.GetCount(key) > 1 {
			continue
		}
		flightStatus := flightStatusInfo.FlightStatus
		flights, hasValue := storage.FlightStorage().GetFlights(int32(flightStatus.AirlineID), flightStatus.FlightNumber)
		isCodeshare := false
		// For now let's believe no flight in the world is partially code-shared
		if hasValue {
			for _, legFlights := range flights {
				for _, segment := range legFlights {
					if segment.IsCodeshare {
						isCodeshare = true
						break
					}
				}
			}
		}
		if isCodeshare {
			continue
		}
		// This should not happen, because MarketingCarrier == 0 implies LegNumber==-1 and we skip dop flights,
		// but just in case
		if flightStatusInfo.FlightPattern.MarketingCarrier == 0 {
			logger.Logger().Warn("Unexpected flights status", log.Reflect("flight_status", flightStatus))
			continue
		}
		// TODO(u-jeen): make delays and cancellation counts proper when the national version is specified in the ban rule
		flightDate := flightStatus.FlightDate
		if len(flightDate) > 10 {
			flightDate = flightDate[:10]
		}
		if blacklistRuleStorage.IsBanned(
			flightStatusInfo.FlightBase, &flightStatusInfo.FlightPattern, dtutil.StringDate(flightDate), "") {
			continue
		}
		statusStorage.UpdateFlightDelays(flightStatus, flightStatusInfo.ScheduledDeparture, flightStatusInfo.ScheduledArrival)
	}

	statusStorage.UpdateDivertedIDs(storage.Stations())

	err = storage.CarriersPopularityScores().UpdateFlightNumbersCache(newDopFlightsStorage)
	if err != nil {
		return nil, exitErrorf("Error while building dop flight numbers popularity cache", err)
	}

	storage.SetStatusStorage(statusStorage)
	storage.SetStationStatusSourceStorage(stationStatusSourceStorage)
	storage.SetBlacklistRuleStorage(blacklistRuleStorage)
	storage.SetFlightMergeRuleStorage(flightMergeRuleStorage)
	storage.FlightStorage().SetDopFlights(newDopFlightsStorage)
	logger.Logger().Info(
		"Loaded flight merge rules",
		log.Int("flight_merge_rules_loaded", flightMergeRuleStorage.RulesCount()),
	)
	logger.Logger().Info(
		"Loaded flight statuses and dop_flights",
		log.Int("statuses_loaded", statusLoader.flightStatusesLoadedCount),
		log.Int("statuses_skipped", statusLoader.flightStatusesSkipped),
		log.Int("dop_flights_created", dopFlightsCreatedCount),
		log.Int("dop_flights_multileg", dopFlightsMultilegCount),
	)
	for key, value := range statusLoader.errorCounters {
		logger.Logger().Warn("Status processing errors", log.Int(key, value))
	}

	return storage, nil
}

func scheduledTime(departureDate string, dayShift int, timeScheduledInt int32, tz *time.Location) string {
	departureDateInt := dtutil.StringDate(departureDate).ToIntDate().AddDays(dayShift)
	if departureDateInt == 0 {
		return ""
	}
	timeValue := dtutil.IntToTime(departureDateInt, dtutil.IntTime(timeScheduledInt), tz)
	if timeValue.IsZero() {
		return ""
	}
	return dtutil.FormatDateTimeISO(timeValue)
}

func getFlightStatusKey(flightStatus structs.FlightStatus) string {
	return fmt.Sprintf(
		"%v.%v.%v.%v",
		flightStatus.AirlineID,
		flightStatus.FlightNumber,
		flightStatus.LegNumber,
		flightStatus.FlightDate,
	)
}

func getFlightStations(flightBase *structs.FlightBase) string {
	return fmt.Sprintf("%v.%v", flightBase.DepartureStation, flightBase.ArrivalStation)
}

type stationByID interface {
	ByID(id int64) (*snapshots.TStationWithCodes, bool)
}

func CreateDopFlight(
	flightStatus structs.FlightStatus,
	dopFlightID int32,
	tzutil timezone.TimeZoneProvider,
	stations stationByID,
) (structs.FlightBase, structs.FlightPattern, bool) {
	if flightStatus.DepartureStation == flightStatus.ArrivalStation {
		return structs.FlightBase{}, structs.FlightPattern{}, false
	}

	departureTz := tzutil.GetTimeZoneByStationID(flightStatus.DepartureStation)
	arrivalTz := tzutil.GetTimeZoneByStationID(flightStatus.ArrivalStation)

	if departureTz == nil {
		logger.Logger().Warn(
			"Can't get timezone for the dop flight departure station",
			log.Int64("departure station", flightStatus.DepartureStation),
		)
		return structs.FlightBase{}, structs.FlightPattern{}, false
	}

	if arrivalTz == nil {
		logger.Logger().Warn(
			"Can't get timezone for the dop flight arrival station",
			log.Int64("arrival station", flightStatus.ArrivalStation),
		)
		return structs.FlightBase{}, structs.FlightPattern{}, false
	}

	departureTimeScheduled := time.Time{}
	arrivalTimeScheduled := time.Time{}
	var err error

	if len(flightStatus.DepartureTimeScheduled) > 0 {
		departureTimeScheduled, err = dtutil.ParseDateTimeISO(flightStatus.DepartureTimeScheduled, departureTz)
		if err != nil {
			logger.Logger().Error(
				"Unable to parse departure date",
				log.Reflect("flight_status", flightStatus),
			)
			departureTimeScheduled = time.Time{}
		}
	}

	if len(flightStatus.ArrivalTimeScheduled) > 0 {
		arrivalTimeScheduled, err = dtutil.ParseDateTimeISO(flightStatus.ArrivalTimeScheduled, arrivalTz)
		if err != nil {
			logger.Logger().Error(
				"Unable to parse arrival date",
				log.Reflect("flight_status", flightStatus),
			)
			arrivalTimeScheduled = time.Time{}
		}
	}

	isHalfFlight := departureTimeScheduled.IsZero() || arrivalTimeScheduled.IsZero()
	if !departureTimeScheduled.IsZero() || !arrivalTimeScheduled.IsZero() {
		flightBase := structs.FlightBase{}
		flightBase.ID = dopFlightID
		flightBase.OperatingCarrier = int32(flightStatus.AirlineID)
		flightBase.OperatingCarrierCode = flightStatus.CarrierCode
		flightBase.OperatingFlightNumber = flightStatus.FlightNumber
		flightBase.LegNumber = 1
		flightBase.Source = structs.GetSource(ssim.EFlightBaseSource_TYPE_DOP)

		flightBase.DepartureStation = flightStatus.DepartureStation
		depStation, depStationOk := stations.ByID(flightStatus.DepartureStation)
		if depStationOk {
			flightBase.DepartureStationCode = depStation.IataCode
		}
		if departureTimeScheduled.IsZero() {
			flightBase.DepartureTimeScheduled = dtutil.TimeNotSpecified
		} else {
			flightBase.DepartureTimeScheduled = int32(dtutil.TimeToIntTime(departureTimeScheduled))
		}
		flightBase.DepartureTerminal = flightStatus.DepartureTerminal

		flightBase.ArrivalStation = flightStatus.ArrivalStation
		arrStation, arrStationOk := stations.ByID(flightStatus.ArrivalStation)
		if arrStationOk {
			flightBase.ArrivalStationCode = arrStation.IataCode
		}
		if arrivalTimeScheduled.IsZero() {
			flightBase.ArrivalTimeScheduled = dtutil.TimeNotSpecified
		} else {
			flightBase.ArrivalTimeScheduled = int32(dtutil.TimeToIntTime(arrivalTimeScheduled))
		}
		flightBase.ArrivalTerminal = flightStatus.ArrivalTerminal

		flightPattern := structs.FlightPattern{}
		flightPattern.ID = dopFlightID
		flightPattern.OperatingFlightPatternID = dopFlightID
		flightPattern.FlightBaseID = flightBase.ID
		flightPattern.LegNumber = flightBase.LegNumber
		if !departureTimeScheduled.IsZero() {
			flightPattern.OperatingFromDate = string(dtutil.FormatDateIso(departureTimeScheduled))
			flightPattern.OperatingUntilDate = flightPattern.OperatingFromDate
			flightPattern.OperatingOnDays = getNonZeroWeekDay(departureTimeScheduled)
		} else if !arrivalTimeScheduled.IsZero() {
			flightPattern.OperatingFromDate = string(dtutil.FormatDateIso(arrivalTimeScheduled))
			flightPattern.OperatingUntilDate = flightPattern.OperatingFromDate
			flightPattern.OperatingOnDays = getNonZeroWeekDay(arrivalTimeScheduled)
		}

		flightPattern.MarketingCarrier = flightBase.OperatingCarrier
		flightPattern.MarketingCarrierCode = flightBase.OperatingCarrierCode
		flightPattern.MarketingFlightNumber = flightBase.OperatingFlightNumber
		flightPattern.IsDop = true
		flightPattern.IsHalfFlight = isHalfFlight
		// TODO(RASPTICKETS-20270): properly handle the case when arrival preceeds departure
		if !arrivalTimeScheduled.IsZero() && !departureTimeScheduled.IsZero() {
			departureDateIndex, departureOk := dtutil.DateCache.IndexOfStringDate(dtutil.FormatDateIso(departureTimeScheduled))
			arrivalDateIndex, arrivalOk := dtutil.DateCache.IndexOfStringDate(dtutil.FormatDateIso(arrivalTimeScheduled))
			if departureOk && arrivalOk {
				flightPattern.ArrivalDayShift = int32(arrivalDateIndex - departureDateIndex)
				if flightPattern.ArrivalDayShift < -1 || flightPattern.ArrivalDayShift > 2 {
					logger.Logger().Error(
						"Unexpected day shift",
						log.Int32("day_shift", flightPattern.ArrivalDayShift),
						log.Reflect("flight_pattern", flightPattern),
					)
					flightPattern.ArrivalDayShift = 0
				}
			}
		}
		return flightBase, flightPattern, true
	}
	return structs.FlightBase{}, structs.FlightPattern{}, false
}

func getNonZeroWeekDay(t time.Time) int32 {
	result := int32(t.Weekday())
	if result != 0 {
		return result
	}
	return 7
}

func writeMetrics(fileName string, counter int, size uint64) {
	switch fileName {
	case "flight_bases.pb2.bin":
		aviaMetrics.GlobalMetrics().Gauge(metrics.LoaderPrefix, metrics.FlightBaseCount).Set(float64(counter))
		aviaMetrics.GlobalMetrics().Gauge(metrics.LoaderPrefix, metrics.FlightBaseSize).Set(float64(size))
	case "flight_patterns.pb2.bin":
		aviaMetrics.GlobalMetrics().Gauge(metrics.LoaderPrefix, metrics.FlightPatternCount).Set(float64(counter))
		aviaMetrics.GlobalMetrics().Gauge(metrics.LoaderPrefix, metrics.FlightPatternSize).Set(float64(size))
	case "flight_statuses.pb2.bin":
		aviaMetrics.GlobalMetrics().Gauge(metrics.LoaderPrefix, metrics.FlightStatusCount).Set(float64(counter))
		aviaMetrics.GlobalMetrics().Gauge(metrics.LoaderPrefix, metrics.FlightStatusSize).Set(float64(size))
	case "carriers.pb2.bin":
		aviaMetrics.GlobalMetrics().Gauge(metrics.LoaderPrefix, metrics.CarriersCount).Set(float64(counter))
		aviaMetrics.GlobalMetrics().Gauge(metrics.LoaderPrefix, metrics.CarriersSize).Set(float64(size))
	case "stations_with_codes.pb2.bin":
		aviaMetrics.GlobalMetrics().Gauge(metrics.LoaderPrefix, metrics.StationsCount).Set(float64(counter))
		aviaMetrics.GlobalMetrics().Gauge(metrics.LoaderPrefix, metrics.StationsSize).Set(float64(size))
	}
}

func exitErrorf(msg string, err error) error {
	logger.Logger().Error(msg, log.Error(err))
	return xerrors.Errorf("%v. Cause: %w", msg, err)
}

func path(parts ...string) string {
	return strings.Join(parts, "/")
}

func progressWriter(
	message string, filename string, aftereach int, messageWriter func(msg string, fields ...log.Field),
) func(int) {
	totalBytes := 0
	lastPrinted := 0
	return func(readBytes int) {
		totalBytes += readBytes
		if totalBytes-lastPrinted > aftereach {
			lastPrinted = totalBytes
			messageWriter(message+" "+filename, log.Int("total", totalBytes))
		}
	}
}
