package accessors

import (
	"fmt"
	"sort"
	"strconv"
	"strings"

	"a.yandex-team.ru/library/go/core/xerrors"
	"a.yandex-team.ru/travel/avia/shared_flights/api/internal/storage"
	"a.yandex-team.ru/travel/avia/shared_flights/api/internal/storage/station"
	"a.yandex-team.ru/travel/avia/shared_flights/api/pkg/structs"
	"a.yandex-team.ru/travel/avia/shared_flights/lib/go/dtutil"
	"a.yandex-team.ru/travel/proto/shared_flights/snapshots"
)

type AccessorSerivce interface {
	GetFlightBase(id int32, isDopFlight bool) (flightBase structs.FlightBase, err error)
	GetFlightPattern(id int32, isDopFlight bool) (flightPattern *structs.FlightPattern, err error)
	GetFlightStatus(bucketKey string, date dtutil.IntDate) (flightStatus *structs.FlightStatus, err error)
	GetIataCorrectionRules() ([]*snapshots.TIataCorrectionRule, error)
	GetRawFlight(marketingCarrier int32, flightNumber string, isDop bool) (flights []FlightPatternAndBase, err error)
	GetRawP2PFlights(from, to int64) ([]FlightP2P, error)
	GetCarrierCodeByID(id int32) string
	GetCarriersByCode(iata string) []int32
	GetFlightStatusSources() []structs.FlightStatusSource
	GetFlightMergeRules() []structs.FlightMergeRule
	GetCarriersPopularityScores(nationalVersion string) map[int32]int32
	GetPopularCarriersForFlight(flightNumber, nationalVersion string) []int32
	Stations() station.StationStorage
}

func NewAccessorService(storage *storage.Storage) AccessorSerivce {
	return &accessorServiceImpl{storage: storage}
}

type accessorServiceImpl struct {
	storage *storage.Storage
}

type FlightPatternAndBase struct {
	FlightPattern structs.FlightPattern `json:"fp"`
	FlightBase    structs.FlightBase    `json:"fb"`
	Err           error                 `json:"error,omitempty"`
}

type FlightP2P struct {
	FlightTitle string   `json:"flight"`
	CarrierIata string   `json:"carrierIata"`
	Route       []string `json:"route"`
}

// For diagnostics only: accessor methods that retrieve the raw data from the storage
func (service *accessorServiceImpl) GetFlightBase(id int32, isDopFlight bool) (flightBase structs.FlightBase, err error) {
	value, err := service.storage.FlightStorage().GetFlightBase(id, isDopFlight)
	if err != nil {
		return value, err
	}
	return value, nil
}

func (service *accessorServiceImpl) GetFlightPattern(id int32, isDopFlight bool) (flightPattern *structs.FlightPattern, err error) {
	if isDopFlight {
		value, hasValue := service.storage.FlightStorage().GetDopFlightPatterns()[id]
		if !hasValue {
			return value, xerrors.Errorf("unknown flight pattern id %v", id)
		}
		return value, nil
	}
	value, hasValue := service.storage.FlightStorage().GetFlightPatterns()[id]
	if !hasValue {
		return value, xerrors.Errorf("unknown flight pattern id %v", id)
	}
	return value, nil
}

func (service *accessorServiceImpl) GetFlightStatus(bucketKey string, date dtutil.IntDate) (flightStatus *structs.FlightStatus, err error) {
	statusesBucketKey := strings.ReplaceAll(bucketKey, "_", ".")
	flightStatuses, hasValue := service.storage.StatusStorage().GetFlightStatuses(statusesBucketKey)
	if !hasValue {
		return nil, xerrors.Errorf("unknown flight bucket %v", bucketKey)
	}

	flightStatus, hasDate := flightStatuses.GetStatus(date)
	if !hasDate {
		var dates []dtutil.IntDate
		for d := range flightStatuses.Keys() {
			dates = append(dates, d)
		}
		return flightStatus, xerrors.Errorf("unknown date %v, available dates: %v", date, dates)
	}

	return flightStatus, nil
}

func (service *accessorServiceImpl) GetRawFlight(
	marketingCarrier int32,
	flightNumber string,
	isDop bool,
) (flights []FlightPatternAndBase, err error) {
	flightPatterns, hasValue := service.storage.FlightStorage().GetFlights(marketingCarrier, flightNumber)
	result := make([]FlightPatternAndBase, 0, len(flightPatterns))
	if !hasValue {
		return result, xerrors.Errorf("unknown flight key %v %v", marketingCarrier, flightNumber)
	}
	for _, legFlights := range flightPatterns {
		for _, fp := range legFlights {
			if fp.IsDop && !isDop {
				continue
			}
			entry := FlightPatternAndBase{FlightPattern: *fp}
			fb, err := service.storage.FlightStorage().GetFlightBase(fp.FlightBaseID, fp.IsDop)
			if err == nil {
				entry.FlightBase = fb
			} else {
				entry.Err = err
			}
			result = append(result, entry)
		}
	}
	return result, nil
}

func (service *accessorServiceImpl) GetRawP2PFlights(from, to int64) ([]FlightP2P, error) {
	flights, hasValue := service.storage.FlightStorage().GetFlightsP2P(from, to)
	if !hasValue {
		return []FlightP2P{}, nil
	}
	var result []FlightP2P
	for _, flight := range flights {
		flightPatterns, ok := service.storage.FlightStorage().GetFlightsByKey(flight)
		iata := ""
		stations := make([]string, 0)
		if ok {
			for _, legFlights := range flightPatterns {
				for _, fp := range legFlights {
					if iata == "" {
						iata = fp.MarketingCarrierCode
					}
					fb, err := service.storage.FlightStorage().GetFlightBase(fp.FlightBaseID, fp.IsDop)
					if err != nil {
						return result, err
					}
					stations = appendStations(
						fb.LegNumber, fb.DepartureStation, fb.DepartureStationCode, fb.ArrivalStation, fb.ArrivalStationCode, stations)
				}
			}
		}

		result = append(
			result,
			FlightP2P{
				FlightTitle: flight,
				CarrierIata: iata,
				Route:       stations,
			},
		)
	}
	return result, nil
}

func (service *accessorServiceImpl) GetIataCorrectionRules() ([]*snapshots.TIataCorrectionRule, error) {
	return service.storage.IataCorrector().GetRules(), nil
}

func (service *accessorServiceImpl) GetFlightStatusSources() []structs.FlightStatusSource {
	sources := service.storage.StatusStorage().GetStatusSources()
	result := make([]structs.FlightStatusSource, 0, len(sources))
	for _, source := range sources {
		result = append(result, source)
	}
	sort.Slice(result, func(i, j int) bool {
		return result[i].Priority > result[j].Priority
	})
	return result
}

func (service *accessorServiceImpl) GetFlightMergeRules() []structs.FlightMergeRule {
	rulesStorage := service.storage.FlightMergeRuleStorage()
	result := rulesStorage.GetRules()
	sort.Slice(result, func(i, j int) bool {
		marketingCarrierDiff := result[i].MarketingCarrier - result[j].MarketingCarrier
		if marketingCarrierDiff != 0 {
			return marketingCarrierDiff < 0
		}
		operatingCarrierDiff := result[i].OperatingCarrier - result[j].OperatingCarrier
		if operatingCarrierDiff != 0 {
			return operatingCarrierDiff < 0
		}
		if result[i].MarketingFlightRegexp != result[j].MarketingFlightRegexp {
			return result[i].MarketingFlightRegexp < result[j].MarketingFlightRegexp
		}
		return result[i].OperatingFlightRegexp < result[j].OperatingFlightRegexp
	})
	return result
}

func (service *accessorServiceImpl) GetCarriersPopularityScores(nationalVersion string) map[int32]int32 {
	return service.storage.CarriersPopularityScores().GetPopularityScores(nationalVersion)
}

func (service *accessorServiceImpl) GetPopularCarriersForFlight(flightNumber, nationalVersion string) []int32 {
	return service.storage.CarriersPopularityScores().GetCarriers(flightNumber, nationalVersion)
}

func appendStations(leg int32, from int64, fromIata string, to int64, toIata string, stations []string) []string {
	for len(stations) < int(leg) {
		stations = append(stations, "")
	}
	legStations := fmt.Sprintf("%v-%v", getStation(fromIata, from), getStation(toIata, to))
	stations[leg-1] = appendStation(stations[leg-1], legStations)
	return stations
}

func getStation(stationIata string, station int64) string {
	value := stationIata
	if value == "" {
		value = strconv.FormatInt(station, 10)
	}
	return value
}

func appendStation(currentValue, newStations string) string {
	if !strings.Contains(currentValue, newStations) {
		if currentValue != "" {
			currentValue = currentValue + ", "
		}
		currentValue = currentValue + newStations
	}
	return currentValue
}

func (service *accessorServiceImpl) GetCarrierCodeByID(id int32) string {
	return service.storage.CarrierStorage().GetCarrierCodeByID(id)
}

func (service *accessorServiceImpl) GetCarriersByCode(iata string) []int32 {
	return service.storage.CarrierStorage().GetCarriersByCode(iata)
}

func (service *accessorServiceImpl) Stations() station.StationStorage {
	return service.storage.Stations()
}
